package fi.kela.kanta.ptayhteiset.liiketoimintalogiikka.util;

import java.time.LocalDateTime;
import java.util.ArrayList;
import java.util.List;
import java.util.Map;

import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import fi.kela.kanta.ptayhteiset.repositorio.metatiedot.Metadata;
import fi.kela.kanta.ptayhteiset.tietomalli.poikkeukset.AppException;
import fi.kela.kanta.ptayhteiset.tietomalli.poikkeukset.ErrorMessages;
import fi.kela.kanta.ptayhteiset.tietomalli.tietotyypit.CVValue;
import fi.kela.kanta.ptayhteiset.tietomalli.vakiot.PTAYhteisetConstants;

/**
 * Apuluokka sanoman tietojen käsittelyyn
 */
public class SanomaAvustaja {

    private static final Logger LOG = LoggerFactory.getLogger(SanomaAvustaja.class);

    private static final String VIRHEELLINEN_AIKALEIMA = "Virheellinen päivämäärä";


    /**
     * Vertaa vastaako metatietojen arvot toisiaan.
     * 
     * @param firstMetadata vertailtava metadata
     * @param secondMetadata vertailtava metadata
     * @return true jos arvot vastaavat toisiaan, muuten false.
     */
    public static boolean compareMetadataValues(Metadata firstMetadata, Metadata secondMetadata) {

	LOG.trace(LogUtils.LOG_BEGIN);

	boolean isEquals = false;

	String firstValue = "";
	String secondValue = "";

	if (firstMetadata != null && firstMetadata.getAttributeValue() != null) {
	    firstValue = firstMetadata.getAttributeValue();
	}

	if (secondMetadata != null && secondMetadata.getAttributeValue() != null) {
	    secondValue = secondMetadata.getAttributeValue();
	}

	if (firstValue.equals(secondValue)) {
	    isEquals = true;
	}

	LOG.trace(LogUtils.LOG_END);

	return isEquals;
    }


    /**
     * Palauttaa aikaleiman muotoiltuna.
     * 
     * @param asiakirjaMetatiedot asiakirjan metatiedot
     * @param attributeName aikaleimaattribuutin nimi
     * @return aikaleima muotoiltuna.
     */
    public static String getFormattedAikaleima(Map<String, Metadata> asiakirjaMetatiedot, String attributeName) {

	LOG.trace(LogUtils.LOG_BEGIN);

	String aikaleima = SanomaAvustaja.getAttributeValue(attributeName, asiakirjaMetatiedot);
	String formattedAikaleima = null;

	if (!aikaleima.isEmpty()) {
	    LocalDateTime dateTime = LocalDateTime.parse(aikaleima, PTArkistoDateTimeConstants.ORACLE_DATE_LONG_JAVA_FORMATTER);
	    formattedAikaleima = PTArkistoDateTime.getDateStrByPattern(dateTime, PTArkistoDateTimeConstants.YYYYMMDDHHMMSS_DATE_PATTERN);

	    if (formattedAikaleima == null) {
		LOG.error(VIRHEELLINEN_AIKALEIMA.concat(PTAYhteisetConstants.FORMATTING_ANCHOR), aikaleima);
		throw new AppException(VIRHEELLINEN_AIKALEIMA, ErrorMessages.ERROR_CODE_2001);
	    }
	}

	LOG.trace(LogUtils.LOG_END);

	return formattedAikaleima;
    }


    /**
     * Palauttaa metadata-objektin attribuutin arvon.
     * 
     * @param metadata metadata-objekti
     * @return attribuutin arvo tai null mikäli metadata-objekti on null
     */
    public static String getMetadataAttributeValue(Metadata metadata) {

	LOG.trace(LogUtils.LOG_BEGIN);

	String attributeValue = null;
	if (metadata != null) {
	    attributeValue = metadata.getAttributeValue();
	}

	LOG.trace(LogUtils.LOG_END);

	return attributeValue;
    }


    /**
     * Palauttaa attribuutin arvon metadatamapista.
     * 
     * @param attributeName attribuutin nimi
     * @param metatiedot metadatamappi
     * @return attribuutin arvo tai tyhjä merkkijono jos attribuutin arvo oli null.
     */
    public static String getAttributeValue(String attributeName, Map<String, Metadata> metatiedot) {

	LOG.trace(LogUtils.LOG_BEGIN);

	String attributeValue = null;

	if (metatiedot.containsKey(attributeName)) {
	    attributeValue = metatiedot.get(attributeName).getAttributeValue();
	}

	LOG.trace(LogUtils.LOG_END);

	return (attributeValue == null ? "" : attributeValue);
    }


    /**
     * Palauttaa attribuutin arvolistan metadatamapista.
     * 
     * @param attributeName attribuutin nimi
     * @param metatiedot metadatamappi
     * @return attribuutin arvolista tai tyhjä lista jos attribuutin arvo oli null.
     */
    public static List<String> getAttributeValues(String attributeName, Map<String, Metadata> metatiedot) {

	LOG.trace(LogUtils.LOG_BEGIN);

	List<String> attributeValues = null;

	if (metatiedot.containsKey(attributeName)) {
	    attributeValues = metatiedot.get(attributeName).getAttributeValues();
	}

	LOG.trace(LogUtils.LOG_END);

	return (attributeValues == null ? new ArrayList<>() : attributeValues);
    }


    /**
     * Palauttaa koodiston ja koodiarvon.
     * 
     * @param asiakirjaMetatiedot asiakirjan metatiedot
     * @param attributeName attribuutin nimi
     * @param defaultCodeSystem oletus koodistoarvo, joka asetetaan mikäli metatieto ei sisällä sitä
     * @return koodiston ja koodiarvon CVValuena.
     */
    public static CVValue getCVValue(Map<String, Metadata> asiakirjaMetatiedot, String attributeName, String defaultCodeSystem) {

	LOG.trace(LogUtils.LOG_BEGIN);

	CVValue cvValue = new CVValue();

	String attributeValue = SanomaAvustaja.getAttributeValue(attributeName, asiakirjaMetatiedot);

	if (!attributeValue.isEmpty()) {

	    String[] values = attributeValue.split(PTAYhteisetConstants.CV_SEPARATOR);

	    if (values.length == 1) {
		cvValue.setCodeSystem(defaultCodeSystem);
		cvValue.setCode(values[0]);
	    } else if (values.length == 2) {
		cvValue.setCodeSystem(values[0]);
		cvValue.setCode(values[1]);
	    }

	}

	LOG.trace(LogUtils.LOG_END);

	return cvValue;
    }


    /**
     * Palauttaa moniarvoisesta metatiedosta parsitun CVValue-listan.
     * 
     * @param asiakirjaMetatiedot asiakirjan metatiedot
     * @param attributeName attribuutin nimi
     * @param defaultCodeSystem oletus koodistoarvo, joka asetetaan mikäli metatieto ei sisällä sitä
     * @param listSeparator erotinmerkki moniarvoisen metatiedon erotinmerkki
     * @param valueSeparator erotinmerkki yksittäisen CV arvon erotinmerkki
     * @return CVValue-lista.
     */
    public static List<CVValue> getCVValueList(Map<String, Metadata> asiakirjaMetatiedot, String attributeName, String defaultCodeSystem, String listSeparator, String valueSeparator) {

	LOG.debug(LogUtils.LOG_BEGIN);

	List<CVValue> cvValueList = new ArrayList<>();

	String attributeValue = getAttributeValue(attributeName, asiakirjaMetatiedot);
	List<String> valueList = TietomallinKasittelyHelper.convertMultiValueStringToList(attributeValue, listSeparator);

	for (String strValue : valueList) {

	    String[] value = strValue.split(valueSeparator);
	    CVValue cvValue = new CVValue();

	    if (value.length == 1 && defaultCodeSystem != null) {
		cvValue.setCodeSystem(defaultCodeSystem);
		cvValue.setCode(value[0]);
		cvValueList.add(cvValue);

	    } else if (value.length == 2) {
		cvValue.setCodeSystem(value[0]);
		cvValue.setCode(value[1]);
		cvValueList.add(cvValue);
	    }
	}

	LOG.debug(LogUtils.LOG_END);

	return cvValueList;
    }


    /**
     * Privaatti konstruktori
     */
    private SanomaAvustaja() {
	throw new IllegalStateException("Utility class");
    }
}
