package fi.kela.kanta.ptayhteiset.liiketoimintalogiikka.util;

import java.io.ByteArrayInputStream;
import java.io.ByteArrayOutputStream;
import java.io.IOException;
import java.io.InputStream;
import java.util.Date;
import java.util.Properties;

import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import fi.kela.kanta.ptayhteiset.tietomalli.poikkeukset.AppException;
import fi.kela.kanta.ptayhteiset.tietomalli.poikkeukset.ErrorMessages;
import fi.kela.kanta.ptayhteiset.tietomalli.vakiot.PTAYhteisetConstants;
import jakarta.activation.DataHandler;
import jakarta.activation.DataSource;
import jakarta.mail.MessagingException;
import jakarta.mail.Multipart;
import jakarta.mail.Part;
import jakarta.mail.Session;
import jakarta.mail.internet.MimeBodyPart;
import jakarta.mail.internet.MimeMessage;
import jakarta.mail.internet.MimeMultipart;
import jakarta.mail.util.ByteArrayDataSource;

/**
 * Mimen käsittelyyn oleva apuluokka
 */
public class MimeHelper {

    private static final Logger LOG = LoggerFactory.getLogger(MimeHelper.class);

    private static final String MIMEVIRHE_DECODE = "Mime dekoodausvirhe";
    private static final String MIMEVIRHE_ENCODE = "Mime enkoodausvirhe";
    private static final String PARAMETRI_NULL_TAI_TYHJA = "Asetettu parametri oli null tai tyhjä";

    /**
     * Purkaa Mime-paketin ja palauttaa tuloksen merkkijonona.
     * 
     * @param encodedMime purettava teksti
     * @return tulos purettuna merkkijonona
     */
    public static String getMimeDecodedValue(String encodedMime) {

        LOG.debug(LogUtils.LOG_BEGIN);

	String decodedMime = "";
	ByteArrayInputStream bais = new ByteArrayInputStream(encodedMime.getBytes());
	InputStream baiss = null;

        try (bais) {

            Properties props = new Properties();
            Session session = Session.getInstance(props, null);

            MimeMessage message = new MimeMessage(session, bais);
            Object objContent = message.getContent();

            if (objContent instanceof String) {
                decodedMime = (String) message.getContent();

            } else if (objContent instanceof Multipart) {

                Multipart multipart = (Multipart) message.getContent();

                if (multipart.getCount() == 0) {
                    throw new AppException(MIMEVIRHE_DECODE, ErrorMessages.ERROR_CODE_1010);
                }

		for (int i = 0, y = multipart.getCount(); i < y; i++) {
		    Part part = multipart.getBodyPart(i);
		    String disposition = part.getDisposition();
		    if (disposition == null) {
			MimeBodyPart mbp = (MimeBodyPart) part;
			baiss = mbp.getInputStream();
			byte[] b = new byte[baiss.available()];
			int read = baiss.read(b);
			decodedMime = new String(b, 0, read);
			baiss.close();
		    }
		}
	    }

	} catch (MessagingException | IOException | IndexOutOfBoundsException e) {

	    LOG.error(MIMEVIRHE_DECODE, e);
	    throw new AppException(MIMEVIRHE_DECODE, e, ErrorMessages.ERROR_CODE_1010);
	}
	finally {
	    
	    // Suljetaan inputStream jos jäänyt sulkematta dekoodausvirheen takia
	    if (baiss!=null) {
		try {
		    baiss.close();
		} catch (IOException e) {
		    // 
		}
	    }
	}

        LOG.debug(LogUtils.LOG_END);

        return decodedMime;
    }


    /**
     * Suorittaa merkkijonon enkoodauksen MIMEksi. Jos suorituksessa tapahtuu virhe, heitetään {@link AppException}
     * 
     * @param sourceText lähdeteksti
     * @return enkoodattu merkkijono
     */
    public static String encodeToMIME(String sourceText) {

        LOG.trace(LogUtils.LOG_BEGIN);

        String msgWithHeader = "";

        if (sourceText == null || sourceText.isEmpty()) {
            LOG.error(PARAMETRI_NULL_TAI_TYHJA);
            throw new AppException(ErrorMessages.ERROR_CODE_1003);
        }

        ByteArrayOutputStream bos = new ByteArrayOutputStream();

        try (bos) {

            Properties props = new Properties();
            Session session = Session.getInstance(props, null);
            MimeMessage message = new MimeMessage(session);
            MimeBodyPart messageBodyPart = new MimeBodyPart();

            messageBodyPart.setText(sourceText, PTAYhteisetConstants.UTF8);
            messageBodyPart.addHeader("Content-transfer-encoding", "base64");
            messageBodyPart.addHeader("Content-type", "text/xml; charset=\"UTF-8\"");

            Multipart multipart = new MimeMultipart("related");
            multipart.addBodyPart(messageBodyPart);

            message.setContent(multipart);
            message.setSentDate(new Date());
            message.saveChanges();
            message.removeHeader("Message-ID");
            message.writeTo(bos);

            msgWithHeader = bos.toString();

            msgWithHeader = msgWithHeader.replaceFirst("<", "&lt;");
            msgWithHeader = msgWithHeader.replaceFirst(">", "&gt;");

        } catch (Exception e) {

            LOG.error(MIMEVIRHE_ENCODE, e);
            throw new AppException(MIMEVIRHE_ENCODE, e, ErrorMessages.ERROR_CODE_1003);
        }

        LOG.trace(LogUtils.LOG_END);

        return msgWithHeader;
    }


    /**
     * Suorittaa binäärisen tiedon enkoodauksen MIMEksi. Jos suorituksessa tapahtuu virhe, heitetään
     * {@link AppException}
     * 
     * @param bs byte array.
     * @return enkoodattu merkkijono
     */
    public static String encodeByteArrayToMIME(byte[] bs) {

        LOG.debug(LogUtils.LOG_BEGIN);
        String msgWithHeader = "";
        if (bs == null || bs.length == 0) {
            LOG.error(PARAMETRI_NULL_TAI_TYHJA);
            throw new AppException(ErrorMessages.ERROR_CODE_1003);
        }

        ByteArrayOutputStream bos = new ByteArrayOutputStream();

        try (bos) {
            Properties props = new Properties();
            Session session = Session.getInstance(props, null);
            MimeMessage message = new MimeMessage(session);
            MimeBodyPart messageBodyPart = new MimeBodyPart();

            DataSource ds = new ByteArrayDataSource(bs, "application/pdf");
            messageBodyPart.setDataHandler(new DataHandler(ds));
            messageBodyPart.addHeader("Content-transfer-encoding", "base64");
            messageBodyPart.addHeader("Content-type", "application/pdf");

            Multipart multipart = new MimeMultipart("related");
            multipart.addBodyPart(messageBodyPart);
            message.setContent(multipart);
            message.setSentDate(new Date());
            message.saveChanges();
            message.removeHeader("Message-ID");
            message.writeTo(bos);

            msgWithHeader = bos.toString();

            msgWithHeader = msgWithHeader.replaceFirst("<", "&lt;");
            msgWithHeader = msgWithHeader.replaceFirst(">", "&gt;");

        } catch (MessagingException | IOException e) {

            LOG.error(MIMEVIRHE_ENCODE, e);
            throw new AppException(MIMEVIRHE_ENCODE, e, ErrorMessages.ERROR_CODE_1003);
        }

        LOG.debug(LogUtils.LOG_END);

        return msgWithHeader;
    }


    /**
     * Privaatti konstruktori
     */
    private MimeHelper() {
        throw new IllegalStateException("Utility class");
    }

}
