package fi.kela.kanta.ptayhteiset.liiketoimintalogiikka.util;

import java.sql.Timestamp;
import java.text.ParseException;
import java.text.SimpleDateFormat;
import java.time.LocalDate;
import java.time.LocalDateTime;
import java.time.ZoneId;
import java.time.ZonedDateTime;
import java.time.format.DateTimeFormatter;
import java.time.format.DateTimeFormatterBuilder;
import java.time.format.ResolverStyle;
import java.time.temporal.ChronoField;
import java.util.Date;
import java.util.HashMap;
import java.util.Map;
import java.util.regex.Matcher;
import java.util.regex.Pattern;

import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import fi.kela.kanta.ptayhteiset.tietomalli.poikkeukset.AppException;
import fi.kela.kanta.ptayhteiset.tietomalli.poikkeukset.ErrorMessages;
import fi.kela.kanta.ptayhteiset.tietomalli.vakiot.PTAYhteisetConstants;

/**
 * Luokka päivämäärien käsittelyyn Kanta-palveluissa.
 */
public class PTArkistoDateTime {

    private static final String VIRHEELLINEN_PVM = "Virheellinen pvm {}";

    private static final Logger LOG = LoggerFactory.getLogger(PTArkistoDateTime.class);

    /**
     * Tuettujen formaattien regular expressionit.
     */
    private static final Map<String, DateTimeFormatter> DATE_FORMATTERS = new HashMap<>();

    private static final Map<String, Pattern> DATE_PATTERNS = new HashMap<>();

    // @formatter:off
    static {
	DATE_FORMATTERS.put(PTArkistoDateTimeConstants.YYYY_DATE_PATTERN,
		 new DateTimeFormatterBuilder().parseStrict().appendPattern(PTArkistoDateTimeConstants.YYYY_DATE_PATTERN)
		.parseDefaulting(ChronoField.DAY_OF_YEAR, 1).parseDefaulting(ChronoField.CLOCK_HOUR_OF_DAY, 24).parseDefaulting(ChronoField.MINUTE_OF_HOUR, 0)
		.toFormatter().withResolverStyle(ResolverStyle.STRICT)
		.withZone(ZoneId.of(PTAYhteisetConstants.EUROPE_HELSINKI)));

	DATE_FORMATTERS.put(PTArkistoDateTimeConstants.YYYYMM_DATE_PATTERN,
		new DateTimeFormatterBuilder().parseStrict().appendPattern(PTArkistoDateTimeConstants.YYYYMM_DATE_PATTERN)
		.parseDefaulting(ChronoField.DAY_OF_MONTH, 1).parseDefaulting(ChronoField.CLOCK_HOUR_OF_DAY, 24).parseDefaulting(ChronoField.MINUTE_OF_HOUR, 0)
		.toFormatter().withResolverStyle(ResolverStyle.STRICT)
		.withZone(ZoneId.of(PTAYhteisetConstants.EUROPE_HELSINKI))
		);

	DATE_FORMATTERS.put(PTArkistoDateTimeConstants.YYYYMMDD_DATE_PATTERN,
		new DateTimeFormatterBuilder().parseStrict().appendPattern(PTArkistoDateTimeConstants.YYYYMMDD_DATE_PATTERN)
		.parseDefaulting(ChronoField.CLOCK_HOUR_OF_DAY, 24).parseDefaulting(ChronoField.MINUTE_OF_HOUR, 0)
		.toFormatter().withResolverStyle(ResolverStyle.STRICT)
		.withZone(ZoneId.of(PTAYhteisetConstants.EUROPE_HELSINKI)));

	DATE_FORMATTERS.put(PTArkistoDateTimeConstants.YYYY_MM_DD_DATE_PATTERN,
		new DateTimeFormatterBuilder().parseStrict().appendPattern(PTArkistoDateTimeConstants.YYYY_MM_DD_DATE_PATTERN)
		.parseDefaulting(ChronoField.CLOCK_HOUR_OF_DAY, 24).parseDefaulting(ChronoField.MINUTE_OF_HOUR, 0)
		.toFormatter().withResolverStyle(ResolverStyle.STRICT)
		.withZone(ZoneId.of(PTAYhteisetConstants.EUROPE_HELSINKI)));

	DATE_FORMATTERS.put(PTArkistoDateTimeConstants.YYYYMMDDHH_DATE_PATTERN,
		DateTimeFormatter.ofPattern(PTArkistoDateTimeConstants.YYYYMMDDHH_DATE_PATTERN).withResolverStyle(ResolverStyle.STRICT)
		.withZone(ZoneId.of(PTAYhteisetConstants.EUROPE_HELSINKI)));
	DATE_FORMATTERS.put(PTArkistoDateTimeConstants.YYYYMMDDHHMM_DATE_PATTERN,
		DateTimeFormatter.ofPattern(PTArkistoDateTimeConstants.YYYYMMDDHHMM_DATE_PATTERN).withResolverStyle(ResolverStyle.STRICT)
		.withZone(ZoneId.of(PTAYhteisetConstants.EUROPE_HELSINKI)));
	DATE_FORMATTERS.put(PTArkistoDateTimeConstants.YYYYMMDDHHMMSS_DATE_PATTERN,
		DateTimeFormatter.ofPattern(PTArkistoDateTimeConstants.YYYYMMDDHHMMSS_DATE_PATTERN).withResolverStyle(ResolverStyle.STRICT)
		.withZone(ZoneId.of(PTAYhteisetConstants.EUROPE_HELSINKI)));
	DATE_FORMATTERS.put(PTArkistoDateTimeConstants.YYYYMMDDHHMMSSZZ_DATE_PATTERN,
		DateTimeFormatter.ofPattern(PTArkistoDateTimeConstants.YYYYMMDDHHMMSSZZ_DATE_PATTERN)
		.withZone(ZoneId.of(PTAYhteisetConstants.EUROPE_HELSINKI)));
	DATE_FORMATTERS.put(PTArkistoDateTimeConstants.ISO_8601_DATE_PATTERN,
		DateTimeFormatter.ofPattern(PTArkistoDateTimeConstants.ISO_8601_DATE_PATTERN)
		.withZone(ZoneId.of(PTAYhteisetConstants.EUROPE_HELSINKI))); // Sama kuin ISO_OFFSET_DATETIME!

	DATE_PATTERNS.put(PTArkistoDateTimeConstants.YYYY_DATE_PATTERN, Pattern.compile(PTArkistoDateTimeConstants.DATE_YYYY_REGEXP));
	DATE_PATTERNS.put(PTArkistoDateTimeConstants.YYYYMM_DATE_PATTERN, Pattern.compile(PTArkistoDateTimeConstants.DATE_YYYYMM_REGEXP));
	DATE_PATTERNS.put(PTArkistoDateTimeConstants.YYYYMMDD_DATE_PATTERN, Pattern.compile(PTArkistoDateTimeConstants.DATE_YYYYMMDD_REGEXP));
	DATE_PATTERNS.put(PTArkistoDateTimeConstants.YYYY_MM_DD_DATE_PATTERN, Pattern.compile(PTArkistoDateTimeConstants.DATE_YYYY_MM_DD_REGEXP));
	DATE_PATTERNS.put(PTArkistoDateTimeConstants.YYYYMMDDHH_DATE_PATTERN, Pattern.compile(PTArkistoDateTimeConstants.DATE_YYYYMMDDHH_REGEXP));
	DATE_PATTERNS.put(PTArkistoDateTimeConstants.YYYYMMDDHHMM_DATE_PATTERN, Pattern.compile(PTArkistoDateTimeConstants.DATE_YYYYMMDDHHMM_REGEXP));
	DATE_PATTERNS.put(PTArkistoDateTimeConstants.YYYYMMDDHHMMSS_DATE_PATTERN, Pattern.compile(PTArkistoDateTimeConstants.DATE_YYYYMMDDHHMMSS_REGEXP));
	DATE_PATTERNS.put(PTArkistoDateTimeConstants.YYYYMMDDHHMMSSZZ_DATE_PATTERN, Pattern.compile(PTArkistoDateTimeConstants.DATE_YYYYMMDDHHMMSSZZ_REGEXP));
	DATE_PATTERNS.put(PTArkistoDateTimeConstants.ISO_8601_DATE_PATTERN, Pattern.compile(PTArkistoDateTimeConstants.ISO_8601_REGEXP)); // Sama kuin ISO_OFFSET_DATETIME!
    }
    // @formatter:on

    /**
     * Yksityinen konstruktori
     */
    private PTArkistoDateTime() {
        // estetään instanssien luonti
    }


    /**
     * Palauttaa annetun päivämäärän halutussa formaatissa
     *
     * @param date päivämäärä
     * @param pattern haluttu formaatti
     * @return päivämäärä merkkijono
     */
    public static String getDateStrByPattern(final LocalDateTime date, final String pattern) {
        return getDateStr(date, pattern);
    }


    /**
     * Palauttaa annetun päivämäärän yyyyMMdd formaatissa
     *
     * @param date päivämäärä
     * @return päivämäärä merkkijono
     */
    public static String getDateStrYYYYMMDD(final LocalDateTime date) {
        return getDateStr(date, PTArkistoDateTimeConstants.YYYYMMDD_DATE_PATTERN);
    }


    /**
     * Palauttaa annetun päivämäärän yyyy-MM-dd-formaatissa
     *
     * @param date päivämäärä
     * @return päivämäärä merkkijonona, esimerkiksi 29-03-2021
     */
    public static String getDateStrYYYY_MM_DD(final LocalDateTime date) {
        return getDateStr(date, PTArkistoDateTimeConstants.YYYY_MM_DD_DATE_PATTERN);
    }


    /**
     * Palauttaa annetun päivämäärän yyyy-MM-dd-formaatissa
     *
     * @param date päivämäärä
     * @return päivämäärä merkkijonona, esimerkiksi 29-03-2021
     */
    public static String getDateStrYYYY_MM_DD(final LocalDate date) {
        return getDateStr(date, PTArkistoDateTimeConstants.YYYY_MM_DD_DATE_PATTERN);
    }


    /**
     * Palauttaa annetun päivämäärän yyyyMMddHH formaatissa
     *
     * @param date päivämäärä
     * @return päivämäärä merkkijono
     */
    public static String getDateStrYYYYMMDDHH(final LocalDateTime date) {
        return getDateStr(date, PTArkistoDateTimeConstants.YYYYMMDDHH_DATE_PATTERN);
    }


    /**
     * Palauttaa annetun päivämäärän yyyyMMddHHmmss formaatissa
     *
     * @param date päivämäärä
     * @return päivämäärä merkkijono
     */
    public static String getDateStrYYYYMMDDHHMMSS(final LocalDateTime date) {
        return getDateStr(date, PTArkistoDateTimeConstants.YYYYMMDDHHMMSS_DATE_PATTERN);
    }


    /**
     * Palauttaa annetun päivämäärän yyyyMMddHHmmssZZ formaatissa
     *
     * @param date päivämäärä
     * @return päivämäärä merkkijono
     */
    public static String getDateStrYYYYMMDDHHMMSSZZ(final LocalDateTime date) {
        return getDateStr(date, PTArkistoDateTimeConstants.YYYYMMDDHHMMSSZZ_DATE_PATTERN);
    }


    /**
     * Palauttaa annetun päivämäärän ISO 8601 formaatissa
     *
     * @param date päivämäärä
     * @return päivämäärä merkkijono
     */
    public static String getDateStrISO8601(final LocalDateTime date) {
        return getDateStr(date, PTArkistoDateTimeConstants.ISO_8601_DATE_PATTERN);
    }


    /**
     * Palauttaa nykyhetken aikaleiman ISO 8601 formaatissa.
     *
     * @return päivämäärä merkkijono
     */
    public static String getCurrentTimeStamp() {
        return getDateStr(LocalDateTime.now(), PTArkistoDateTimeConstants.ISO_8601_DATE_PATTERN);
    }


    /**
     * Muuntaa annetun annetun merkkijonon päivämääräksi jos merkkijono vastaa tuettuja formaatteja.
     *
     * @param dateStr muunnettava aika
     * @return päivämäärä
     */
    public static LocalDateTime getDateTime(final String dateStr) {

        LOG.debug(LogUtils.LOG_BEGIN);

        LocalDateTime date = null;

        if (dateStr != null) {
            DateTimeFormatter formatter = getHL7DateFormat(dateStr);

            if (formatter != null) {
                date = LocalDateTime.from(formatter.parse(dateStr));
            }

        }

        LOG.debug(LogUtils.LOG_END);

        return date;
    }


    /**
     * Muuntaa annetun annetun merkkijonon päivämääräksi jos merkkijono vastaa tuettuja formaatteja.
     *
     * @param dateStr muunnettava aika
     * @return päivämäärä
     */
    public static LocalDate getDate(final String dateStr) {

        LOG.debug(LogUtils.LOG_BEGIN);

        LocalDate date = null;

        if (dateStr != null) {
            DateTimeFormatter formatter = getHL7DateFormat(dateStr);

            if (formatter != null) {
                date = LocalDate.from(formatter.parse(dateStr));
            }

        }

        LOG.debug(LogUtils.LOG_END);

        return date;
    }


    /**
     * Konvertoi merkkijonomuotoisen aikaleimakentän LocalDateTime objektiksi. Jos suorituksessa tapahtuu virhe,
     * heitetään {@link AppException}
     *
     * @param strDate merkkijonomuotoinen aikaleimakenttä
     * @return merkkijonosta konvertoitu LocalDateTime objekti
     */
    public static LocalDateTime stringDateToLocalDateTime(String strDate) {

        LOG.debug(LogUtils.LOG_BEGIN);

        LocalDateTime retDateTime = PTArkistoDateTime.getDateTime(strDate);

        if (strDate != null && !strDate.isEmpty() && retDateTime == null) {

            LOG.error(VIRHEELLINEN_PVM, strDate, ErrorMessages.ERROR_CODE_2001);
            throw new AppException(ErrorMessages.ERROR_CODE_2001);

        }

        LOG.debug(LogUtils.LOG_END);

        return retDateTime;
    }


    /**
     * Konvertoi merkkijonomuotoisen aikaleimakentän LocalDateTime objektiksi, johon on lisätty tunnit, minuutit ja
     * sekunnit päivän loppuun saakka. Jos suorituksessa tapahtuu virhe, heitetään {@link AppException}
     *
     * @param strDate merkkijonomuotoinen aikaleimakenttä
     * @return merkkijonosta konvertoitu LocalDateTime objekti, johon on lisätty tunnit, minuutit ja sekunnit päivän
     *         loppuun saakka.
     */
    public static LocalDateTime stringEndDateToLocalDateTime(String strDate) {

        LOG.debug(LogUtils.LOG_BEGIN);

        if (strDate != null && PTArkistoDateTimeConstants.YYYYMMDD_DATE_PATTERN.length() == strDate.length()) {
            strDate += PTArkistoDateTimeConstants.DATE_HOURS;
        }

        LocalDateTime retDateTime = PTArkistoDateTime.getDateTime(strDate);

        if (strDate != null && !strDate.isEmpty() && retDateTime == null) {

            LOG.error(VIRHEELLINEN_PVM, strDate, ErrorMessages.ERROR_CODE_2001);
            throw new AppException(ErrorMessages.ERROR_CODE_2001);

        }

        LOG.debug(LogUtils.LOG_END);

        return retDateTime;
    }


    /**
     * Konvertoi merkkijonomuotoisen aikaleimakentän LocalDate objektiksi. Jos suorituksessa tapahtuu virhe, heitetään
     * {@link AppException}
     *
     * @param strDate merkkijonomuotoinen aikaleimakenttä
     * @return merkkijonosta konvertoitu LocalDate objekti
     */
    public static LocalDate stringDateToLocalDate(String strDate) {

        LOG.debug(LogUtils.LOG_BEGIN);

        LocalDate date = PTArkistoDateTime.getDate(strDate);

        if (strDate != null && !strDate.isEmpty() && date == null) {

            LOG.error(VIRHEELLINEN_PVM, strDate, ErrorMessages.ERROR_CODE_2001);
            throw new AppException(ErrorMessages.ERROR_CODE_2001);

        }

        LOG.debug(LogUtils.LOG_END);

        return date;

    }


    /**
     * Konvertoi java.sql.Timestamp aikaleiman merkkijonoksi.
     *
     * @param ts konvertoitava aikaleima
     * @return konvertoitu merkkijono
     */
    public static String sqlTimeStampToString(Timestamp ts) {

        LOG.debug(LogUtils.LOG_BEGIN);

        LocalDateTime localDT = null;

        if (ts != null && !PTAYhteisetConstants.getNullTimestamp().equals(ts)) {
            localDT = ts.toInstant().atZone(ZoneId.of(PTAYhteisetConstants.EUROPE_HELSINKI)).toLocalDateTime();
        }

        LOG.debug(LogUtils.LOG_END);

        return (localDT == null) ? null : PTArkistoDateTimeConstants.ORACLE_DATE_LONG_JAVA_FORMATTER.format(localDT);
    }


    /**
     * Muuttaa merkkijonomuodossa olevan paikallisen aikavyöhykkeen aikaleiman UTC aikaleimaksi
     *
     * @param localDTStr paikallisen aikavyöhykkeen aikaleima
     * @param formatter aikaleimamuotoilija
     * @return muutettu aikaleima merkkijonona
     */
    public static String localDateTimeStringToUtcDateTimeString(String localDTStr, DateTimeFormatter formatter) {

        LOG.debug(LogUtils.LOG_BEGIN);

        LocalDateTime ldt = LocalDateTime.from(formatter.parse(localDTStr));
        ZonedDateTime zdtEuropeHelsinki = ZonedDateTime.of(ldt, ZoneId.of(PTAYhteisetConstants.EUROPE_HELSINKI));
        ZonedDateTime zdtUTC = zdtEuropeHelsinki.withZoneSameInstant(ZoneId.of(PTAYhteisetConstants.UTC));

        LOG.debug(LogUtils.LOG_END);

        return formatter.format(zdtUTC);
    }


    /**
     * Konvertoi syötteenä saadun PTAYhteisetConstants.BIRTHDATE_FORMAT) formaatissa olevan aikaleimakentän uudeksi
     * aikaleimaksi käyttäen formaattia PTAYhteisetConstants.ORACLE_DATE_LONG_JAVA_FORMAT, johon on lisätty tunnit,
     * minuutit ja sekunnit päivän alkuhetkeen. Jos suorituksessa tapahtuu virhe, heitetään {@link AppException}
     *
     * @param strDate merkkijonomuotoinen aikaleimakenttä
     * @return merkkijonosta konvertoitu aikaleima, johon on lisätty tunnit, minuutit ja sekunnit kyseisen päivän
     *         alkuhetkeen.
     */
    public static String stringDateToStartDateTimeString(String strDate) {

        LOG.debug(LogUtils.LOG_BEGIN);

        String dateOfBirth = null;

        try {

            if (strDate == null || strDate.length() != PTAYhteisetConstants.BIRTHDATE_FORMAT.length()) {
                LOG.error(VIRHEELLINEN_PVM, strDate, ErrorMessages.ERROR_CODE_2001);
                throw new AppException(ErrorMessages.ERROR_CODE_2001);
            }

            SimpleDateFormat birthDateFormat = new SimpleDateFormat(PTAYhteisetConstants.BIRTHDATE_FORMAT);
            birthDateFormat.setLenient(false);
            Date patientBirthTimeDate = birthDateFormat.parse(strDate);

            SimpleDateFormat oracleDateFormat = new SimpleDateFormat(PTAYhteisetConstants.ORACLE_DATE_LONG_JAVA_FORMAT);
            oracleDateFormat.setLenient(false);
            dateOfBirth = oracleDateFormat.format(patientBirthTimeDate);

        } catch (ParseException e) {
            LOG.error(VIRHEELLINEN_PVM, strDate, ErrorMessages.ERROR_CODE_2001);
            throw new AppException(e, ErrorMessages.ERROR_CODE_2001);
        }

        LOG.debug(LogUtils.LOG_END);

        return dateOfBirth;
    }


    /**
     * Palauttaa HL7 date - merkkijonoa vastaavan formatterin.
     *
     * @param dateStr päivämäärä merkkijonona
     * @return päivämäärää vastaava formaatti.
     */
    public static DateTimeFormatter getHL7DateFormat(final String dateStr) {

        LOG.debug(LogUtils.LOG_BEGIN);

        String pattern = null;
        DateTimeFormatter format = null;

        switch (dateStr.length()) {

            case 4:
                pattern = PTArkistoDateTimeConstants.YYYY_DATE_PATTERN;
                break;
            case 6:
                pattern = PTArkistoDateTimeConstants.YYYYMM_DATE_PATTERN;
                break;
            case 8:
                pattern = PTArkistoDateTimeConstants.YYYYMMDD_DATE_PATTERN;
                break;
            case 10:
                if (dateStr.contains("-")) {
                    pattern = PTArkistoDateTimeConstants.YYYY_MM_DD_DATE_PATTERN;
                } else {
                    pattern = PTArkistoDateTimeConstants.YYYYMMDDHH_DATE_PATTERN;
                }
                break;
            case 12:
                pattern = PTArkistoDateTimeConstants.YYYYMMDDHHMM_DATE_PATTERN;
                break;
            case 14:
                pattern = PTArkistoDateTimeConstants.YYYYMMDDHHMMSS_DATE_PATTERN;
                break;
            case 18:
                pattern = PTArkistoDateTimeConstants.YYYYMMDDHHMMSSZZ_DATE_PATTERN;
                break;
            default:
                pattern = null;
        }

        if (pattern != null) {
            Matcher m = DATE_PATTERNS.get(pattern).matcher(dateStr);
            if (m.matches()) {
                format = DATE_FORMATTERS.get(pattern);
            }
        }

        LOG.debug(LogUtils.LOG_END);

        return format;
    }


    /**
     * Palauttaa annetun päivämäärän annetussa formaatissa
     *
     * @param date päivämäärä
     * @param pattern haluttu formaatti
     * @return päivämäärä merkkijono
     */
    private static String getDateStr(final LocalDateTime date, final String pattern) {

        LOG.debug(LogUtils.LOG_BEGIN);

        String dateStr = (date != null && pattern != null) ? DateTimeFormatter.ofPattern(pattern).format(date) : null;

        LOG.debug(LogUtils.LOG_END);

        return dateStr;
    }


    /**
     * Palauttaa annetun päivämäärän annetussa formaatissa
     *
     * @param date päivämäärä
     * @param pattern haluttu formaatti
     * @return päivämäärä merkkijono
     */
    private static String getDateStr(final LocalDate date, final String pattern) {

        LOG.debug(LogUtils.LOG_BEGIN);

        String dateStr = (date != null && pattern != null) ? DateTimeFormatter.ofPattern(pattern).format(date) : null;

        LOG.debug(LogUtils.LOG_END);

        return dateStr;
    }
}
