package fi.kela.kanta.ptayhteiset.liiketoimintalogiikka.xml;

import java.io.StringReader;
import java.util.ArrayList;
import java.util.Collections;
import java.util.List;

import javax.xml.transform.sax.SAXSource;
import javax.xml.xpath.XPath;
import javax.xml.xpath.XPathConstants;
import javax.xml.xpath.XPathExpression;
import javax.xml.xpath.XPathExpressionException;
import javax.xml.xpath.XPathFactory;

import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.xml.sax.InputSource;

import fi.kela.kanta.ptayhteiset.liiketoimintalogiikka.util.LogUtils;
import fi.kela.kanta.ptayhteiset.liiketoimintalogiikka.util.TietomallinKasittelyHelper;
import fi.kela.kanta.ptayhteiset.tietomalli.poikkeukset.AppException;
import fi.kela.kanta.ptayhteiset.tietomalli.poikkeukset.ErrorMessages;
import fi.kela.kanta.ptayhteiset.tietomalli.tietotyypit.CV;
import fi.kela.kanta.ptayhteiset.tietomalli.tietotyypit.CVValue;
import fi.kela.kanta.ptayhteiset.tietomalli.tietotyypit.IIValue;
import fi.kela.kanta.ptayhteiset.tietomalli.tietotyypit.NullFlavor;
import fi.kela.kanta.ptayhteiset.tietomalli.tietotyypit.PN;
import fi.kela.kanta.ptayhteiset.tietomalli.vakiot.PTAYhteisetConstants;
import net.sf.saxon.om.NodeInfo;
import net.sf.saxon.om.TreeInfo;
import net.sf.saxon.trans.XPathException;

/**
 * Saxon-kirjastoa käyttävä toteutus XML:n käsittelyyn.
 */
public class SaxonXPathHelper implements IXPathHelper {

    private static final Logger LOG = LoggerFactory.getLogger(SaxonXPathHelper.class);
    private static final String VIRHE_SANOMAN_KASITTELYSSA = "Virhe sanoman käsittelyssä: {}";
    private static final String VIRHEELLINEN_PARAMETRI = "Virheellinen parametri";

    private TreeInfo document = null;
    private XPath xpathObj = null;

    /**
     * SaxonXPathHelper luokan konstruktori. Jos suorituksessa tapahtuu virhe, heitetään {@link AppException}
     * 
     * @param xmlString xml-muotoinen merkkijono
     */
    public SaxonXPathHelper(String xmlString) {

        LOG.debug(LogUtils.LOG_BEGIN);

        if (xmlString == null || xmlString.isEmpty()) {

            LOG.error(VIRHEELLINEN_PARAMETRI.concat(PTAYhteisetConstants.FORMATTING_ANCHOR),
                    ErrorMessages.ERROR_CODE_1017);
            throw new AppException(VIRHEELLINEN_PARAMETRI, ErrorMessages.ERROR_CODE_1017);

        }

        final StringReader xmlReader = new StringReader(xmlString);

        try (xmlReader) {

            XPathFactory xpFactory = new net.sf.saxon.xpath.XPathFactoryImpl();

            xpathObj = xpFactory.newXPath();
            xpathObj.setNamespaceContext(new HL7NamespaceContext());

            InputSource inputSource = new InputSource(xmlReader);
            SAXSource saxSrc = new SAXSource(inputSource);
            net.sf.saxon.Configuration config = ((net.sf.saxon.xpath.XPathFactoryImpl) xpFactory).getConfiguration();

            document = config.buildDocumentTree(saxSrc);

        } catch (XPathException e) {
            LOG.error(VIRHE_SANOMAN_KASITTELYSSA, ErrorMessages.ERROR_CODE_2010);
            throw new AppException(e, ErrorMessages.ERROR_CODE_2010);
        }

        LOG.debug(LogUtils.LOG_END);
    }


    /**
     * {@inheritDoc} Jos suorituksessa tapahtuu virhe, heitetään {@link AppException}
     */
    @Override
    public String getStringValue(String expression) {

        LOG.trace(LogUtils.LOG_BEGIN);

        String retVal = "";
        XPathExpression processCmd;

        try {

            processCmd = xpathObj.compile(expression);
            String code = (String) processCmd.evaluate(document, XPathConstants.STRING);

            if (code != null) {
                retVal = code;
            }

        } catch (XPathExpressionException e) {
            LOG.error(VIRHE_SANOMAN_KASITTELYSSA, ErrorMessages.ERROR_CODE_2010);
            throw new AppException(e, ErrorMessages.ERROR_CODE_2010);
        }

        LOG.trace(LogUtils.LOG_END);

        return retVal;
    }


    /**
     * {@inheritDoc} Jos suorituksessa tapahtuu virhe, heitetään {@link AppException}
     */
    @Override
    public IIValue getIIValue(String expression) {

        LOG.trace(LogUtils.LOG_BEGIN);

        IIValue iIValue = new IIValue();
        iIValue.setRoot(getStringValue(expression + "/@root"));
        iIValue.setExtension(getStringValue(expression + "/@extension"));

        LOG.trace(LogUtils.LOG_END);

        return iIValue;
    }


    /**
     * {@inheritDoc} Jos suorituksessa tapahtuu virhe, heitetään {@link AppException}
     */
    @Override
    public IIValue getIIValueWithNullFlavor(String expression) {

        LOG.trace(LogUtils.LOG_BEGIN);

        IIValue iIValue = new IIValue();
        iIValue.setRoot(getStringValue(expression + "/@root"));
        iIValue.setExtension(getStringValue(expression + "/@extension"));

        if ((getStringValue(expression + "/@nullFlavor")).equals(NullFlavor.NA.getValue())) {

            iIValue.setNullflavor(NullFlavor.NA);
        }

        LOG.trace(LogUtils.LOG_END);

        return iIValue;
    }


    /**
     * {@inheritDoc} Jos suorituksessa tapahtuu virhe, heitetään {@link AppException}
     */
    @Override
    public String getIIStringValue(String expression) {

        LOG.trace(LogUtils.LOG_BEGIN);

        String retValue = getIIStringValue(expression, PTAYhteisetConstants.SEPARATOR);

        LOG.trace(LogUtils.LOG_END);

        return retValue;
    }


    /**
     * {@inheritDoc} Jos suorituksessa tapahtuu virhe, heitetään {@link AppException}
     */
    @Override
    public String getIIStringValue(String expression, String separator) {

        LOG.trace(LogUtils.LOG_BEGIN);

        String retValue = "";
        String root = getStringValue(expression + "/@root");
        String extension = getStringValue(expression + "/@extension");

        if (!root.isEmpty() && !extension.isEmpty()) {
            retValue = new StringBuilder().append(root).append(separator).append(extension).toString();
        } else if (!root.isEmpty()) {
            retValue = root;
        }

        LOG.trace(LogUtils.LOG_END);

        return retValue;
    }


    /**
     * {@inheritDoc} Jos suorituksessa tapahtuu virhe, heitetään {@link AppException}
     */
    @Override
    public CVValue getCVValue(String expression) {

        LOG.trace(LogUtils.LOG_BEGIN);

        CVValue code = new CVValue();

        code.setCodeSystem(getStringValue(expression + "/@codeSystem"));
        code.setCode(getStringValue(expression + "/@code"));

        LOG.trace(LogUtils.LOG_END);

        return code;
    }

    /**
     * {@inheritDoc} Jos suorituksessa tapahtuu virhe, heitetään {@link AppException}
     */
    @Override
    public CV getCV(String expression) {

        LOG.trace(LogUtils.LOG_BEGIN);

        CV code = new CV();

        code.setCodeSystem(getStringValue(expression + "/@codeSystem"));
        code.setCode(getStringValue(expression + "/@code"));
        code.setCodeSystemName(getStringValue(expression + "/@codeSystemName"));
        code.setDisplayName(getStringValue(expression + "/@displayName"));
        code.setNullFlavor(getStringValue(expression + "/@nullFlavor"));

        LOG.trace(LogUtils.LOG_END);

        return code;
    }


    /**
     * {@inheritDoc} Jos suorituksessa tapahtuu virhe, heitetään {@link AppException}
     */
    @Override
    public String getCVStringValue(String expression) {

        LOG.trace(LogUtils.LOG_BEGIN);

        String retValue = "";

        String codeSystem = getStringValue(expression + "/@codeSystem");
        String code = getStringValue(expression + "/@code");

        if (!codeSystem.isEmpty() && !code.isEmpty()) {
            retValue = new StringBuilder().append(codeSystem).append(PTAYhteisetConstants.CV_SEPARATOR).append(code)
                    .toString();
        }

        LOG.trace(LogUtils.LOG_END);

        return retValue;
    }


    /**
     * {@inheritDoc} Jos suorituksessa tapahtuu virhe, heitetään {@link AppException}
     */
    @Override
    public List<String> getStringValuesFromNodeList(String expression) {

        LOG.trace(LogUtils.LOG_BEGIN);

        List<String> retVal = new ArrayList<>();

        try {

            XPathExpression processCmd = xpathObj.compile(expression);

            @SuppressWarnings("unchecked")
            List<NodeInfo> values = (List<NodeInfo>) processCmd.evaluate(document, XPathConstants.NODESET);

            if (values != null) {
                for (NodeInfo nodeInfo : values) {
                    retVal.add(nodeInfo.getStringValue());
                }
            }

        } catch (XPathExpressionException e) {
            LOG.error(VIRHE_SANOMAN_KASITTELYSSA, ErrorMessages.ERROR_CODE_2010);
            throw new AppException(e, ErrorMessages.ERROR_CODE_2010);
        }

        LOG.trace(LogUtils.LOG_END);

        return retVal;
    }


    /**
     * {@inheritDoc} Jos suorituksessa tapahtuu virhe, heitetään {@link AppException}
     */
    @Override
    public boolean checkIfNodeExists(String expression) {

        LOG.trace(LogUtils.LOG_BEGIN);

        boolean matches = false;

        try {

            XPathExpression processCmd = xpathObj.compile(expression);

            @SuppressWarnings("unchecked")
            List<NodeInfo> values = (List<NodeInfo>) processCmd.evaluate(document, XPathConstants.NODESET);

            if (values != null && !values.isEmpty()) {
                matches = true;
            }

        } catch (XPathExpressionException e) {
            LOG.error(VIRHE_SANOMAN_KASITTELYSSA, ErrorMessages.ERROR_CODE_2010);
            throw new AppException(e, ErrorMessages.ERROR_CODE_2010);
        }

        LOG.trace(LogUtils.LOG_END);

        return matches;
    }


    /**
     * {@inheritDoc} Jos suorituksessa tapahtuu virhe, heitetään {@link AppException}
     */
    @Override
    public List<String> getIIValuesAsStringFromNodeList(String expression) {
        return getIIValuesAsStringFromNodeList(expression, PTAYhteisetConstants.SEPARATOR);
    }


    /**
     * {@inheritDoc} Jos suorituksessa tapahtuu virhe, heitetään {@link AppException}
     */
    public List<String> getIIValuesAsStringFromNodeList(String expression, String separator) {

        LOG.trace(LogUtils.LOG_BEGIN);

        if (separator == null) {
            separator = PTAYhteisetConstants.SEPARATOR;
        }

        List<String> retVal = new ArrayList<>();

        try {

            XPathExpression processCmd = xpathObj.compile(expression);

            @SuppressWarnings("unchecked")
            List<NodeInfo> values = (List<NodeInfo>) processCmd.evaluate(document, XPathConstants.NODESET);

            if (values != null) {
                for (NodeInfo nodeInfo : values) {

                    String root = nodeInfo.getAttributeValue("", "root");
                    String extension = nodeInfo.getAttributeValue("", "extension");

                    LOG.trace("root {}", root);
                    LOG.trace("extension {}", extension);

                    if ((root != null && !root.isEmpty()) && (extension != null && !extension.isEmpty())) {
                        retVal.add(new StringBuilder().append(root).append(separator).append(extension).toString());
                    } else if (root != null && !root.isEmpty()) {
                        retVal.add(root);
                    } else {
                        retVal.add("");
                    }
                }
            }

        } catch (XPathExpressionException e) {
            LOG.error(VIRHE_SANOMAN_KASITTELYSSA, ErrorMessages.ERROR_CODE_2010);
            throw new AppException(e, ErrorMessages.ERROR_CODE_2010);
        }

        LOG.trace(LogUtils.LOG_END);

        return retVal;
    }


    /**
     * {@inheritDoc} Jos suorituksessa tapahtuu virhe, heitetään {@link AppException}
     */
    @Override
    public List<String> getUniqueIIValuesAsStringFromNodeList(String expression) {

        LOG.trace(LogUtils.LOG_BEGIN);

        List<String> retVal = TietomallinKasittelyHelper.removeDuplicates(getIIValuesAsStringFromNodeList(expression));

        retVal.removeIf(value -> (value == null || value.isEmpty()));

        LOG.trace(LogUtils.LOG_END);

        return retVal;
    }


    /**
     * {@inheritDoc} Jos suorituksessa tapahtuu virhe, heitetään {@link AppException}
     */
    @Override
    public List<IIValue> getIIValuesFromNodeList(String expression) {
        return getIIValuesFromNodeList(expression, true);
    }


    /**
     * {@inheritDoc} Jos suorituksessa tapahtuu virhe, heitetään {@link AppException}
     */
    @Override
    public List<IIValue> getUniqueIIValuesFromNodeList(String expression) {
        return getIIValuesFromNodeList(expression, false);
    }


    /**
     * {@inheritDoc} Jos suorituksessa tapahtuu virhe, heitetään {@link AppException}
     */
    @Override
    public List<String> getCVValuesAsStringFromNodeList(String expression) {
        LOG.trace(LogUtils.LOG_BEGIN);

        List<String> retVal = new ArrayList<>();

        try {

            XPathExpression processCmd = xpathObj.compile(expression);

            @SuppressWarnings("unchecked")
            List<NodeInfo> values = (List<NodeInfo>) processCmd.evaluate(document, XPathConstants.NODESET);

            if (values != null) {
                for (NodeInfo nodeInfo : values) {

                    String codeSystem = nodeInfo.getAttributeValue("", "codeSystem");
                    String code = nodeInfo.getAttributeValue("", "code");

                    LOG.trace("codeSystem {}", codeSystem);
                    LOG.trace("code {}", code);

                    if ((codeSystem != null && !codeSystem.isEmpty()) && (code != null && !code.isEmpty())) {
                        retVal.add(new StringBuilder().append(codeSystem).append(PTAYhteisetConstants.CV_SEPARATOR)
                                .append(code).toString());
                    } else {
                        retVal.add("");
                    }
                }
            }

        } catch (XPathExpressionException e) {
            LOG.error(VIRHE_SANOMAN_KASITTELYSSA, ErrorMessages.ERROR_CODE_2010);
            throw new AppException(e, ErrorMessages.ERROR_CODE_2010);
        }

        LOG.trace(LogUtils.LOG_END);

        return retVal;
    }


    /**
     * {@inheritDoc} Jos suorituksessa tapahtuu virhe, heitetään {@link AppException}
     */
    @Override
    public List<String> getUniqueCVValuesAsStringFromNodeList(String expression) {
        LOG.trace(LogUtils.LOG_BEGIN);

        List<String> retVal = TietomallinKasittelyHelper.removeDuplicates(getCVValuesAsStringFromNodeList(expression));

        retVal.removeIf(value -> (value == null || value.isEmpty()));

        LOG.trace(LogUtils.LOG_END);

        return retVal;
    }


    /**
     * {@inheritDoc} Jos suorituksessa tapahtuu virhe, heitetään {@link AppException}
     */
    @Override
    public List<CVValue> getCVValuesFromNodeList(String expression) {
        return getCVValuesFromNodeList(expression, true);
    }


    /**
     * {@inheritDoc} Jos suorituksessa tapahtuu virhe, heitetään {@link AppException}
     */
    @Override
    public List<CVValue> getUniqueCVValuesFromNodeList(String expression) {
        return getCVValuesFromNodeList(expression, false);
    }


    /**
     * Poimii kokoelman CV-tyyppisiä tietoja annetun xpath-lausekkeen perusteella. Xpath-lausekkeen osoittamasta
     * elementistä poimitaan /@codeSystem ja /@code attribuutit. Jos suorituksessa tapahtuu virhe, heitetään
     * AppException
     * 
     * @param expression xpath-lauseke, jonka perusteella arvo etsitään
     * @param allowDuplicate sallitaanko myös duplikaattiarvot
     * @return xpath-lausekkeen tulos CVValuelistana
     */
    private List<CVValue> getCVValuesFromNodeList(String expression, boolean allowDuplicate) {
        LOG.trace(LogUtils.LOG_BEGIN);

        List<CVValue> retVal = new ArrayList<>();

        try {

            XPathExpression processCmd = xpathObj.compile(expression);

            @SuppressWarnings("unchecked")
            List<NodeInfo> values = (List<NodeInfo>) processCmd.evaluate(document, XPathConstants.NODESET);

            if (values != null) {
                for (NodeInfo nodeInfo : values) {

                    CVValue value = new CVValue();

                    value.setCodeSystem(nodeInfo.getAttributeValue("", "codeSystem"));
                    value.setCode(nodeInfo.getAttributeValue("", "code"));

                    LOG.trace("codeSystem {}", value.getCodeSystem());
                    LOG.trace("code {}", value.getCode());

                    if (allowDuplicate || !retVal.contains(value)) {
                        retVal.add(value);
                    }

                }
            }

        } catch (XPathExpressionException e) {
            LOG.error(VIRHE_SANOMAN_KASITTELYSSA, ErrorMessages.ERROR_CODE_2010);
            throw new AppException(e, ErrorMessages.ERROR_CODE_2010);
        }

        LOG.trace(LogUtils.LOG_END);

        return retVal;
    }


    /**
     * Poimii kokoelman II-tyyppisiä tietoja annetun xpath-lausekkeen perusteella. Xpath-lausekkeen osoittamasta
     * elementistä poimitaan /@root ja /@extension attribuutit. Palauttaa arvot II-tyyppisinä objekteina.
     * 
     * @param expression xpath-lauseke, jonka perusteella arvo etsitään
     * @param allowDuplicate sallitaanko myös duplikaattiarvot
     * @return xpath-lausekkeen tulos IIValue-listana
     */
    public List<IIValue> getIIValuesFromNodeList(String expression, boolean allowDuplicate) {

        LOG.trace(LogUtils.LOG_BEGIN);

        List<IIValue> retVal = new ArrayList<>();

        try {

            XPathExpression processCmd = xpathObj.compile(expression);

            @SuppressWarnings("unchecked")
            List<NodeInfo> values = (List<NodeInfo>) processCmd.evaluate(document, XPathConstants.NODESET);

            if (values != null) {
                for (NodeInfo nodeInfo : values) {

                    IIValue value = new IIValue();
                    String root = nodeInfo.getAttributeValue("", "root");
                    String extension = nodeInfo.getAttributeValue("", "extension");

                    if (root != null && !root.isEmpty()) {
                        value.setRoot(root);
                    } else {
                        value.setRoot("");
                    }

                    if (extension != null && !extension.isEmpty()) {
                        value.setExtension(extension);
                    } else {
                        value.setExtension("");
                    }

                    LOG.trace("root {}", value.getRoot());
                    LOG.trace("extension {}", value.getExtension());

                    if (allowDuplicate || !retVal.contains(value)) {
                        retVal.add(value);
                    }
                }
            }

        } catch (XPathExpressionException e) {
            LOG.error(VIRHE_SANOMAN_KASITTELYSSA, ErrorMessages.ERROR_CODE_2010);
            throw new AppException(e, ErrorMessages.ERROR_CODE_2010);
        }

        LOG.trace(LogUtils.LOG_END);

        return retVal;
    }


    /**
     * {@inheritDoc} Jos suorituksessa tapahtuu virhe, heitetään {@link AppException}
     */
    @Override
    public String getStringValueFromLomakeAsiakirja(int code) {

        try {

            XPathExpression processCmd = xpathObj
                    .compile(String.format(XPathExpressionHakuConstants.LOMAKE_VALUE, code));

            return (String) processCmd.evaluate(document, XPathConstants.STRING);

        } catch (XPathExpressionException e) {
            LOG.error(VIRHE_SANOMAN_KASITTELYSSA, ErrorMessages.ERROR_CODE_2010);
            throw new AppException(e, ErrorMessages.ERROR_CODE_2010);
        }

    }


    /**
     * {@inheritDoc} Jos suorituksessa tapahtuu virhe, heitetään {@link AppException}
     */
    @Override
    public String getSTValueFromLomakeAsiakirja(int code) {

        try {

            XPathExpression processCmd = xpathObj
                    .compile(String.format(XPathExpressionHakuConstants.LOMAKE_ST_VALUE, code));

            return (String) processCmd.evaluate(document, XPathConstants.STRING);

        } catch (XPathExpressionException e) {
            LOG.error(VIRHE_SANOMAN_KASITTELYSSA, ErrorMessages.ERROR_CODE_2010);
            throw new AppException(e, ErrorMessages.ERROR_CODE_2010);
        }
    }


    /**
     * {@inheritDoc} Jos suorituksessa tapahtuu virhe, heitetään {@link AppException}
     */
    @Override
    public String getStringValue(String xpath, String xpath2, int i) {

        StringBuilder xpBuilder = new StringBuilder("(").append(xpath).append(")[");
        xpBuilder.append(i).append("]").append("/").append(xpath2);

        return this.getStringValue(xpBuilder.toString());
    }

    /**
     * {@inheritDoc} Jos suorituksessa tapahtuu virhe, heitetään {@link AppException}
     */
    @Override
    public CV getCV(String xpath, String xpath2, int i) {

        StringBuilder xpBuilder = new StringBuilder("(").append(xpath).append(")[");
        xpBuilder.append(i).append("]").append("/").append(xpath2);

        return this.getCV(xpBuilder.toString());
    }
    /**
     * {@inheritDoc} Jos suorituksessa tapahtuu virhe, heitetään {@link AppException}
     */
    @Override
    public PN getPN(String xpath, String xpath2, int i) {

        StringBuilder xpBuilder = new StringBuilder("(").append(xpath).append(")[");
        xpBuilder.append(i).append("]").append("/").append(xpath2);

        return this.getPN(xpBuilder.toString());
    }
    /**
     * {@inheritDoc} Jos suorituksessa tapahtuu virhe, heitetään {@link AppException}
     */
    @Override
    public String getLomakeKoodi() {

        try {

            XPathExpression processCmd = xpathObj.compile(XPathExpressionHakuConstants.LOMAKE_KOODI);

            return (String) processCmd.evaluate(document, XPathConstants.STRING);

        } catch (XPathExpressionException e) {
            LOG.error(VIRHE_SANOMAN_KASITTELYSSA, ErrorMessages.ERROR_CODE_2010);
            throw new AppException(e, ErrorMessages.ERROR_CODE_2010);
        }
    }


    /**
     * {@inheritDoc} Jos suorituksessa tapahtuu virhe, heitetään {@link AppException}
     */
    @Override
    public int getNodeCount(String expression) {

        try {

            StringBuilder sb = new StringBuilder();
            sb.append("count(").append(expression).append(")");
            XPathExpression processCmd = xpathObj.compile(sb.toString());

            return (int) (double) processCmd.evaluate(document, XPathConstants.NUMBER);

        } catch (XPathExpressionException e) {
            LOG.error(VIRHE_SANOMAN_KASITTELYSSA, ErrorMessages.ERROR_CODE_2010);
            throw new AppException(e, ErrorMessages.ERROR_CODE_2010);
        }
    }


    /**
     * {@inheritDoc} Jos suorituksessa tapahtuu virhe, heitetään {@link AppException}
     */

    @SuppressWarnings("unchecked")
    @Override
    public List<NodeInfo> getNodeList(String expression) {

        try {

            XPathExpression processCmd = xpathObj.compile(expression);

            return (List<NodeInfo>) processCmd.evaluate(document, XPathConstants.NODESET);

        } catch (XPathExpressionException e) {
            LOG.error(VIRHE_SANOMAN_KASITTELYSSA, ErrorMessages.ERROR_CODE_2010);
            throw new AppException(e, ErrorMessages.ERROR_CODE_2010);
        }
    }


    @Override
    public List<String> getTemplateIds() {

        List<String> templateIdt = new ArrayList<>();
        try {

            XPathExpression processCmd = xpathObj.compile(XPathExpressionHakuConstants.TEMPLATE_ID_XPATH);
            @SuppressWarnings("unchecked")
            List<NodeInfo> values = (List<NodeInfo>) processCmd.evaluate(document, XPathConstants.NODESET);
            if (values == null || values.isEmpty()) {
                return Collections.emptyList();
            }
            for (NodeInfo nodeInfo : values) {

                String root = nodeInfo.getAttributeValue("", "root");
                String extension = nodeInfo.getAttributeValue("", "extension");

                if (extension == null || extension.isBlank()) {
                    templateIdt.add(root);
                } else {
                    templateIdt.add(new StringBuilder(root).append('.').append(extension).toString());
                }
            }

        } catch (XPathExpressionException e) {
            LOG.error(VIRHE_SANOMAN_KASITTELYSSA, ErrorMessages.ERROR_CODE_2010);
            throw new AppException(e, ErrorMessages.ERROR_CODE_2010);
        }
        return templateIdt;
    }


    @Override
    public PN getPN(String xpath) {

        PN personName = new PN();

        try {
            XPathExpression processCmd = xpathObj.compile(xpath);
            @SuppressWarnings("unchecked")
            List<NodeInfo> values = (List<NodeInfo>) processCmd.evaluate(document, XPathConstants.NODESET);
            if (values.isEmpty()) {
                return personName;
            }
            int givencount = 0;
            for (NodeInfo nodeInfo : values) {
                String nodeName = nodeInfo.getDisplayName();
                String content = nodeInfo.getStringValue();
                String qualifier = nodeInfo.getAttributeValue("", "qualifier");
                switch (nodeName) {
                    case "given":
                        personName.addName(qualifier, content, givencount, PN.NameType.GIVEN);
                        givencount++;
                        break;
                    case "family":
                        personName.addName(qualifier, content, givencount, PN.NameType.FAMILY);
                        break;
                    case "prefix":
                        personName.addName(qualifier, content, givencount, PN.NameType.PREFIX);
                        break;
                    case "suffix":
                        personName.addName(qualifier, content, givencount, PN.NameType.SUFFIX);
                        break;
                    default:
                        LOG.info("unknown nodetype on name: [{}]", nodeName);
                }
            }

        } catch (XPathExpressionException e) {
            LOG.error(VIRHE_SANOMAN_KASITTELYSSA, ErrorMessages.ERROR_CODE_2010);
            throw new AppException(e, ErrorMessages.ERROR_CODE_2010);
        }
        return personName;
    }
}
