package fi.kela.kanta.ptayhteiset.liiketoimintalogiikka.util;

import fi.kela.kanta.ptayhteiset.tietomalli.tietotyypit.IIValue;
import fi.kela.kanta.ptayhteiset.tietomalli.vakiot.PTAYhteisetConstants;
import jakarta.validation.constraints.NotNull;
import org.apache.commons.lang3.StringUtils;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import java.time.LocalDate;
import java.util.List;

/**
 * Yleisiä päättelysääntöjä tukevat helper-metodit sisältävä apuluokka
 */
public class LiiketoimintaHelper {

    private static final Logger LOG = LoggerFactory.getLogger(LiiketoimintaHelper.class);

    /**
     * Palauttaa rekisteritarkenteen muotoiltuna.
     *
     * @param value rekisteritarkenne
     * @return rekisteritarkenne muotoiltuna tai tyhjä merkkijono jos annettu parametri oli null.
     */
    public static String getFormattedRekisteriTarkenne(IIValue value) {

	LOG.trace(LogUtils.LOG_BEGIN);

	String patientRegistrySpecifier = "";

	if (value != null) {

	    String root = (value.getRoot() == null) ? "" : value.getRoot();
	    String extension = (value.getExtension() == null) ? "" : value.getExtension();
	    patientRegistrySpecifier = value.toString();

	    if (!PTAYhteisetConstants.PATIENTID_OFFICIAL_CODEBASE.equals(root)) {

		StringBuilder valueBuff = new StringBuilder(root);

		if (extension.startsWith("0")) {
		    extension = extension.substring(1);
		}
		extension = extension.replace("-", "");

		if (!extension.isEmpty()) {
		    valueBuff.append(PTAYhteisetConstants.SEPARATOR).append(extension);
		}

		patientRegistrySpecifier = valueBuff.toString();
	    }
	}

	LOG.trace(LogUtils.LOG_END);

	return patientRegistrySpecifier;
    }


    /**
     * Tulkitsee onko annettu hetu virallinen rootin arvon ja extensionin perusteella. Ei tarkista annettuja arvoja
     * hetutarkistuskaavan mukaisesti.
     *
     * @param patientId hetu
     * @return true jos virallinen hetu, muuten false
     */
    public static boolean isOfficialHetu(IIValue patientId) {

	LOG.trace(LogUtils.LOG_BEGIN);

	boolean isOfficial = patientId != null && patientId.isValid() && PTAYhteisetConstants.PATIENTID_OFFICIAL_CODEBASE.equals(
			patientId.getRoot()) && patientId.toString().length() == PTAYhteisetConstants.PATIENTID_OFFICIAL_LENGTH;

	LOG.trace(LogUtils.LOG_END);

	return isOfficial;
    }


    /**
     * Tulkitsee onko annettu hetu virallinen. Ei tarkista annettuja arvoja hetutarkistuskaavan mukaisesti.
     *
     * @param patientId hetu
     * @return true jos virallinen hetu, muuten false
     */
    public static boolean isOfficialHetu(String patientId) {

	LOG.trace(LogUtils.LOG_BEGIN);

	boolean isOfficial = patientId != null && patientId.length() == PTAYhteisetConstants.PATIENTID_OFFICIAL_LENGTH && patientId.startsWith(
			PTAYhteisetConstants.PATIENTID_OFFICIAL_CODEBASE.concat(PTAYhteisetConstants.SEPARATOR));

	LOG.trace(LogUtils.LOG_END);

	return isOfficial;
    }


    /**
     * Palauttaa annetun syntymäajan perusteella tiedon onko potilas täysi-ikäinen.
     *
     * @param birthTime asiakirjasta poimittu syntymäaika
     * @return true jos täysi-ikäinen (18v), muuten false
     */
    public static boolean isAdult(String birthTime) {

	LOG.trace(LogUtils.LOG_BEGIN);

	boolean retVal = false;

	LocalDate todayLD = LocalDate.now();
	LocalDate birthDateLD = PTArkistoDateTime.stringDateToLocalDate(birthTime);

	if (birthDateLD != null) {
	    retVal = birthDateLD.plusYears(18).isBefore(todayLD) || birthDateLD.plusYears(18).equals(todayLD);
	}

	LOG.trace(LogUtils.LOG_FORMAT_ONE_PARAM_END, retVal);

	return retVal;
    }


    /**
     * Tarkastaa potilaan henkilötunnuksen oikeellisuuden
     *
     * @param hetu Tarkastettava hetu
     * @return true jos hetu virallinen ja oikean muotoinen, muissa tapauksissa false, ml. virhetilanteissa.
     */
    public static boolean validatePatientId(String hetu) {

	LOG.trace(LogUtils.LOG_BEGIN);

	if (StringUtils.isNotBlank(hetu)) {
	    // Tarkastetaan hetun pituus
	    if (hetu.length() != PTAYhteisetConstants.PATIENTID_EXTENSION_OFFICIAL_LENGTH) {

		LOG.trace(LogUtils.LOG_END);
		return false;
	    }
	    // Tarkastetaan sotun syntaksi
	    if (!PTAYhteisetConstants.SSN_PATTERN.matcher(hetu).matches()) {

		LOG.trace(LogUtils.LOG_END);
		return false;
	    }

	    // Poimitaan syntymäaika ja tarkastusnumero
	    int number = 0;
	    try {
		number = Integer.parseInt(hetu.substring(0, 6) + hetu.substring(7, 10));
	    } catch (NumberFormatException ex) {

		LOG.trace(LogUtils.LOG_END);
		return false;
	    }

	    // Tarkastetaan että tarkastusnumero on oikea
	    int leftOver = number % 31;
	    String check = PTAYhteisetConstants.CHECKTABLE[leftOver];
	    if (!check.equalsIgnoreCase(hetu.substring(10, 11))) {

		LOG.trace(LogUtils.LOG_END);
		return false;
	    }
	} else {

	    LOG.trace(LogUtils.LOG_END);
	    return false;
	}

	LOG.trace(LogUtils.LOG_END);
	return true;
    }


    /**
     * Palauttaa tiedon vastaako annettu merkkijono OID-patternia.
     *
     * @param oid tarkastettava merkkijono
     * @return true, jos merkkijono vastaa OID-patternia, muuten false
     */
    public static boolean isOid(String oid) {

	LOG.debug(LogUtils.LOG_BEGIN);

	boolean isOid = false;

	if (oid != null && !oid.isEmpty()) {

	    isOid = PTAYhteisetConstants.OID_PATTERN.matcher(oid).matches();
	}

	LOG.debug(LogUtils.LOG_END);

	return isOid;
    }


    /**
     * Muodostaa annetusta merkkijonosta IIValuen, jossa oid-muotoinen osa on sijoitettu root-kenttään ja loppuosa
     * extension-kenttään.
     *
     * @param strOid parsittava merkkijono
     * @return merkkijonosta muodostettu IIValue
     */
    public static IIValue convertToIIValue(String strOid) {

	LOG.debug(LogUtils.LOG_BEGIN);

	IIValue iiValue = new IIValue();

	if (strOid != null && !strOid.isBlank()) {

	    String oidPart = getOidPart(strOid);

	    if (strOid.length() > oidPart.length()) {

		if (!oidPart.isEmpty()) { // ainakin osa on oid-muotoista

		    iiValue.setRoot(oidPart); // oid-muotoinen rootiin
		    iiValue.setExtension(strOid.substring(oidPart.length() + 1)); // loput extensioniin

		} else { // oid-muotoinen osa tyhjä => kaikki extensioniin
		    iiValue.setExtension(strOid);
		}

	    } else { // kokonaan oid-muotoinen => kaikki rootiin
		iiValue.setRoot(strOid);
	    }
	}

	LOG.debug(LogUtils.LOG_END);

	return iiValue;
    }


    /**
     * Palauttaa potilaan virallisesta henkilötunnuksesta muodostetun syntymäajan.
     *
     * @param patientId potilaan virallinen henkilötunnus
     * @return potilaan virallisesta henkilötunnuksesta muodostettu syntymäaika tai null mikäli syntymäaikaa ei saada
     *         muodostettua.
     */
    public static String getBirthTimeFromPatientId(String patientId) {

	LOG.debug(LogUtils.LOG_BEGIN);

	StringBuilder birthTimefromPatientID = new StringBuilder();

	if (validatePatientId(patientId)) {

		final List<String> century20Indicators = List.of("A", "B", "C", "D", "E", "F");
		final List<String> century19Indicators = List.of("-", "Y", "X", "W", "V", "U");
	    String centuryIndicator = patientId.substring(6, 7);

	    if (century19Indicators.contains(centuryIndicator)) {
		birthTimefromPatientID.append("19");
	    } else if (century20Indicators.contains(centuryIndicator)) {
		birthTimefromPatientID.append("20");
	    } else if (centuryIndicator.equals("+")) {
		birthTimefromPatientID.append("18");
	    }

	    birthTimefromPatientID.append(patientId, 4, 6);
	    birthTimefromPatientID.append(patientId, 2, 4);
	    birthTimefromPatientID.append(patientId, 0, 2);
	}

	LOG.debug(LogUtils.LOG_END);

	return birthTimefromPatientID.length() == 8 ? birthTimefromPatientID.toString() : null;
    }


    /**
     * Palauttaa annetusta merkkijonosta oid-muotoisen osuuden.
     *
     * @param strOid parsittava merkkijono
     * @return oid-muotoinen osuus annetusta merkkijonosta tai tyhjän merkkijonon jos se ei ole ollenkaan oid-muotoinen.
     */
    private static String getOidPart(@NotNull String strOid) {

	LOG.debug(LogUtils.LOG_BEGIN);

	StringBuilder sb = new StringBuilder();

	// ensin testataan kokonaisena
	if (!isOid(strOid)) {

	    for (String value : strOid.split("\\.")) {

		sb.append(value);

		// testataan palanen kerrallaan onko edelleen oid-muotoinen
		if (isOid(sb.toString())) {

		    sb.append(PTAYhteisetConstants.SEPARATOR);

		} else {
		    sb.setLength(sb.length() >= (value.length() + 1) ? sb.length() - (value.length() + 1) : 0);
			break;
		}

	    }

	} else {
	    // valmiiksi jo oid, kelpaa sellaisenaan
	    sb.append(strOid);
	}

	LOG.debug(LogUtils.LOG_END);

	return sb.toString();
    }


    /**
     * Privaatti konstruktori
     */
    private LiiketoimintaHelper() {
	throw new IllegalStateException("Utility class");
    }
}
