package fi.kela.kanta.ptayhteiset.liiketoimintalogiikka.xml;

import java.util.List;

import fi.kela.kanta.ptayhteiset.tietomalli.tietotyypit.CV;
import fi.kela.kanta.ptayhteiset.tietomalli.tietotyypit.CVValue;
import fi.kela.kanta.ptayhteiset.tietomalli.tietotyypit.IIValue;
import fi.kela.kanta.ptayhteiset.tietomalli.tietotyypit.PN;
import net.sf.saxon.om.NodeInfo;

/**
 * Julkinen rajapinta XML:n käsittelyyn.
 */
public interface IXPathHelper {

    /**
     * Poimii tiedon xml elementistä annetun xpath-lausekkeen perusteella.
     * 
     * @param expression xpath-lauseke, jonka perusteella arvo etsitään
     * @return xpath-lausekkeen tulos merkkijonona
     */
    public String getStringValue(String expression);


    /**
     * Poimii II-tyyppisen tiedon xml elementistä annetun xpath-lausekkeen perusteella. Lisää xpath-lausekkeeseen /@root
     * ja /@extension attribuutit. Palauttaa IIValue-dataobjektissa root- ja extension -elementeistä poimitut kentät.
     * 
     * @param expression xpath-lauseke, jonka perusteella arvo etsitään ja joka osoittaa II-kenttään
     * @return xpath-lausekkeen tulos IIValue-objektina, jossa on root ja extension-attribuutit
     */
    public IIValue getIIValue(String expression);


    /**
     * Poimii II-tyyppisen tiedon xml elementistä annetun xpath-lausekkeen perusteella. Lisää xpath-lausekkeeseen /@root
     * ja /@extension attribuutit ja käsittelee kentän NullFlavor-arvon. Palauttaa IIValue-dataobjektissa root-,
     * extension- ja nullFlavor -elementeistä poimitut kentät.
     * 
     * @param expression xpath-lauseke, jonka perusteella arvo etsitään ja joka osoittaa II-kenttään
     * @return xpath-lausekkeen tulos IIValue-objektina, jossa on root ja extension-attribuutit
     */
    public IIValue getIIValueWithNullFlavor(String expression);


    /**
     * Poimii II-tyyppisen tiedon xml elementistä annetun xpath-lausekkeen perusteella. Lisää xpath-lausekkeeseen /@root
     * ja /@extension attribuutit. Mikäli sekä /@root että /@extension attribuuteista löytyvät arvot niin ne yhdistetään
     * pisteellä (root.extension). Jos /@extension attribuutissa ei ole arvoa niin palautetaan vain /@root-attribuutin
     * arvo.
     * 
     * @param expression xpath-lauseke, jonka perusteella arvo etsitään
     * @return xpath-lausekkeen tulos merkkijonona
     */
    public String getIIStringValue(String expression);


    /**
     * Poimii II-tyyppisen tiedon xml elementistä annetun xpath-lausekkeen perusteella. Lisää xpath-lausekkeeseen /@root
     * ja /@extension attribuutit. Mikäli sekä /@root että /@extension attribuuteista löytyvät arvot niin ne yhdistetään
     * annetulla erotinmerkillä esim. (root.extension). Jos /@extension attribuutissa ei ole arvoa niin palautetaan vain
     * /@root-attribuutin arvo.
     * 
     * @param expression xpath-lauseke, jonka perusteella arvo etsitään
     * @param separator erotinmerkki jolla @root ja @extension yhdistetään
     * @return xpath-lausekkeen tulos merkkijonona
     */
    public String getIIStringValue(String expression, String separator);


    /**
     * Poimii CV-tyyppisen tiedon xml elementistä annetun xpath-lausekkeen perusteella. Lisää xpath-lausekkeeseen
     * /@codeSystem ja /@code attribuutit.
     * 
     * @param expression xpath-lauseke, jonka perusteella arvo etsitään ja joka osoittaa CV-tyyppiseen kenttään, jolla
     *        on code ja codesystem
     * @return CVValue-objekti, jolle on poimittu XML:stä code ja codesystem
     */

    public CVValue getCVValue(String expression);

    /**
     * Poimii CV-tyyppisen tiedon xml elementistä annetun xpath-lausekkeen perusteella. Lisää xpath-lausekkeeseen
     * /@codeSystem, codeSystemName, displayName, nullFlavor ja /@code attribuutit.
     * 
     * @param expression xpath-lauseke, jonka perusteella arvo etsitään ja joka osoittaa CV/CD/CE-tyyppiseen kenttään, 
     * @return CV-objekti, jolle on poimittu XML:stä kaikki tietotyypin tukemat attribuutit paitsi translation
     */

    public CV getCV(String expression);

    /**
     * Poimii CV-tyyppisen tiedon xml elementistä annetun xpath-lausekkeen perusteella. Lisää xpath-lausekkeeseen
     * /@codeSystem ja /@code attribuutit. Mikäli sekä /@codeSystem että /@code attribuuteista löytyvät arvot niin ne
     * yhdistetään !-erotinmerkillä (codeSystem!code).
     * 
     * @param expression xpath-lauseke, jonka perusteella arvo etsitään
     * @return xpath-lausekkeen tulos merkkijonona
     */
    public String getCVStringValue(String expression);


    /**
     * Poimii kokoelman tietoja annetun xpath-lausekkeen perusteella.
     * 
     * @param expression xpath-lauseke, jonka perusteella arvo etsitään
     * @return xpath-lausekkeen tulos merkkijonolistana
     */
    public List<String> getStringValuesFromNodeList(String expression);


    /**
     * Poimii kokoelman II-tyyppisiä tietoja annetun xpath-lausekkeen perusteella. Xpath-lausekkeen osoittamasta
     * elementistä poimitaan /@root ja /@extension attribuutit. Mikäli sekä /@root että /@extension attribuuteista
     * löytyvät arvot niin ne yhdistetään pisteellä (root.extension) ja lisätään merkkijono palautettavaan listaan. Jos
     * /@extension attribuutissa ei ole arvoa ja /@root attribuutissa on arvo niin palautettavaan listaan lisätään vain
     * /@root-attribuutin arvo. Mikäli elementti löytyy mutta attribuuteissa ei ole arvoa niin lisätään listaan tyhjä
     * merkkijono.
     * 
     * @param expression xpath-lauseke, jonka perusteella arvo etsitään
     * @return xpath-lausekkeen tulos merkkijonolistana
     */
    public List<String> getIIValuesAsStringFromNodeList(String expression);


    /**
     * Poimii kokoelman II-tyyppisiä tietoja annetun xpath-lausekkeen perusteella. Xpath-lausekkeen osoittamasta
     * elementistä poimitaan /@root ja /@extension attribuutit. Mikäli sekä /@root että /@extension attribuuteista
     * löytyvät arvot niin ne yhdistetään annetulla erotinmerkillä esim. (root.extension) ja lisätään merkkijono
     * palautettavaan listaan. Jos /@extension attribuutissa ei ole arvoa ja /@root attribuutissa on arvo niin
     * palautettavaan listaan lisätään vain /@root-attribuutin arvo. Mikäli elementti löytyy mutta attribuuteissa ei ole
     * arvoa niin lisätään listaan tyhjä merkkijono.
     * 
     * @param expression xpath-lauseke, jonka perusteella arvo etsitään
     * @param separator erotinmerkki jolla @root ja @extension yhdistetään
     * @return xpath-lausekkeen tulos merkkijonolistana
     */
    public List<String> getIIValuesAsStringFromNodeList(String expression, String separator);


    /**
     * Poimii kokoelman yksilöllisiä II-tyyppisiä tietoja annetun xpath-lausekkeen perusteella. Xpath-lausekkeen
     * osoittamasta elementistä poimitaan /@root ja /@extension attribuutit. Mikäli sekä /@root että /@extension
     * attribuuteista löytyvät arvot niin ne yhdistetään pisteellä (root.extension) ja lisätään merkkijono
     * palautettavaan listaan. Jos /@extension attribuutissa ei ole arvoa ja /@root attribuutissa on arvo niin
     * palautettavaan listaan lisätään vain /@root-attribuutin arvo. Mikäli elementti löytyy mutta attribuuteissa ei ole
     * arvoa niin listaan ei lisätä tyhjää arvoa.
     * 
     * @param expression xpath-lauseke, jonka perusteella arvo etsitään
     * @return xpath-lausekkeen tulos merkkijonolistana
     */
    public List<String> getUniqueIIValuesAsStringFromNodeList(String expression);


    /**
     * Poimii kokoelman CV-tyyppisiä tietoja annetun xpath-lausekkeen perusteella. Xpath-lausekkeen osoittamasta
     * elementistä poimitaan /@codeSystem ja /@code attribuutit. Mikäli sekä /@codeSystem että /@code attribuuteista
     * löytyvät arvot niin ne yhdistetään !-erotinmerkillä (codeSystem!code) ja lisätään merkkijono palautettavaan
     * listaan. Mikäli elementti löytyy mutta attribuuteissa ei ole arvoa niin lisätään listaan tyhjä merkkijono.
     * 
     * @param expression xpath-lauseke, jonka perusteella arvo etsitään
     * @return xpath-lausekkeen tulos merkkijonolistana
     */
    public List<String> getCVValuesAsStringFromNodeList(String expression);


    /**
     * Poimii kokoelman yksilöllisiä CV-tyyppisiä tietoja annetun xpath-lausekkeen perusteella. Xpath-lausekkeen
     * osoittamasta elementistä poimitaan /@codeSystem ja /@code attribuutit. Mikäli sekä /@codeSystem että /@code
     * attribuuteista löytyvät arvot niin ne yhdistetään !-erotinmerkillä (codeSystem!code) ja lisätään merkkijono
     * palautettavaan listaan. Mikäli elementti löytyy mutta attribuuteissa ei ole arvoa niin listaan ei lisätä tyhjää
     * arvoa.
     * 
     * @param expression xpath-lauseke, jonka perusteella arvo etsitään
     * @return xpath-lausekkeen tulos merkkijonolistana
     */
    public List<String> getUniqueCVValuesAsStringFromNodeList(String expression);


    /**
     * Palauttaa tiedon löytyykö Node tai attribuutti annetun xpath-lausekkeen perusteella.
     * 
     * @param expression xpath-lauseke, jonka perusteella arvo etsitään
     * @return true jos Node tai attribuutti löytyy, muuten false
     */
    public boolean checkIfNodeExists(String expression);


    /**
     * Poimii kokoelman II-tyyppisiä tietoja annetun xpath-lausekkeen perusteella. Xpath-lausekkeen osoittamasta
     * elementistä poimitaan /@root ja /@extension attribuutit. Palauttaa arvot II-tyyppisinä objekteina.
     * 
     * @param expression xpath-lauseke, jonka perusteella arvo etsitään
     * @return xpath-lausekkeen tulos IIValue-listana
     */
    public List<IIValue> getIIValuesFromNodeList(String expression);


    /**
     * Poimii kokoelman yksilöllisiä II-tyyppisiä tietoja annetun xpath-lausekkeen perusteella. Xpath-lausekkeen
     * osoittamasta elementistä poimitaan /@root ja /@extension attribuutit. Palauttaa arvot II-tyyppisinä objekteina.
     * 
     * @param expression xpath-lauseke, jonka perusteella arvo etsitään
     * @return xpath-lausekkeen tulos IIValue-listana
     */
    public List<IIValue> getUniqueIIValuesFromNodeList(String expression);


    /**
     * Poimii kokoelman CV-tyyppisiä tietoja annetun xpath-lausekkeen perusteella. Xpath-lausekkeen osoittamasta
     * elementistä poimitaan /@codeSystem ja /@code attribuutit.
     * 
     * @param expression xpath-lauseke, jonka perusteella arvo etsitään
     * @return xpath-lausekkeen tulos CVValuelistana
     */
    public List<CVValue> getCVValuesFromNodeList(String expression);


    /**
     * Poimii kokoelman yksilöllisiä CV-tyyppisiä tietoja annetun xpath-lausekkeen perusteella. Xpath-lausekkeen
     * osoittamasta elementistä poimitaan /@codeSystem ja /@code attribuutit.
     * 
     * @param expression xpath-lauseke, jonka perusteella arvo etsitään
     * @return xpath-lausekkeen tulos CVValuelistana
     */
    public List<CVValue> getUniqueCVValuesFromNodeList(String expression);


    /**
     * Poimii lomakkeen value-elementissä olevan arvon
     * 
     * @param code kohdistaa haun lomakkeen oikeaan elementtiin.
     * @return value kentässä oleva arvo merkkijonona.
     */
    public String getStringValueFromLomakeAsiakirja(int code);


    /**
     * Hakee ST-tietotyypin arvon, muista poiketen ST ei ole value-elementissä. (ei tarvitse == ei ole?)
     * 
     * @param code lomakekoodi jolla haku kohdistetaan lomakkeen oikeaan elementtiin.
     * @return ST-tietotyypissä olevan merkkijono.
     */
    public String getSTValueFromLomakeAsiakirja(int code);


    /**
     * Palauttaa merkkijonon halutusta indeksistä.
     * 
     * @param xpath xpath jonka indeksiä käytetään.
     * @param xpath2 xpath joka on suora jatko xpath parametrille, eli täydentää kyselypath:in koko xml:n kattavaksi..
     * @param i haettava indeksi
     * @return merkkijono halutusta xpath-polusta.
     */
    public String getStringValue(String xpath, String xpath2, int i);

    /**
     * Palauttaa CV tietotyypin halutusta indeksistä.
     * 
     * @param xpath xpath jonka indeksiä käytetään.
     * @param xpath2 xpath joka on suora jatko xpath parametrille, eli täydentää kyselypath:in koko xml:n kattavaksi..
     * @param i haettava indeksi
     * @return CV halutusta xpath-polusta.
     */
    public CV getCV(String xpath, String xpath2, int i);

    /**
     * Palauttaa nimen PN muodossa halutusta indeksistä.
     * 
     * @param xpath xpath jonka indeksiä käytetään.
     * @param xpath2 xpath joka on suora jatko xpath parametrille, eli täydentää kyselypath:in koko xml:n kattavaksi..
     * @param i haettava indeksi
     * @return PN halutusta xpath-polusta.
     */
    public PN getPN(String xpath, String xpath2, int i);


    /**
     * Hakee cda-asiakirjan lomakekoodin.
     * 
     * @return cda-asiakirjan lomakekoodin merkkijonona
     */
    public String getLomakeKoodi();

    /**
     * Hakee cda-asiakirjan merkintä-tason templateId:t. Jos yhtään templateId:tä ei löydy palautuu tyhjä List
     * Jos merkinnällä ei ole template:id ko. merkinnän kohdalla ei palaudu mitään. Näin ollen templateId.tä voi 
     * palautua vähemmän kuin asiakirjalla on merkintöjä. Jos template:idtä on useita per merkintä menee asia perin 
     * monimutkaiseksi eikä sitä tässä voi selittää.
     * 
     * Jos templateId:ssä on root ja extension palautuu root . extension.
     * (templateId ei sisällä oid-epäyhteensopivia extension-attribuutteja.)
     * 
     * @return cda-asiakirjan merkintöjen templateIdt, jos yhtään templateId:tä ei löydy tyhjä lista.
     */
    public List<String> getTemplateIds();

    /**
     * Poimii henkiön nimen cda-asiakirjasta.
     * @param xpath xpath polku asiakirjan kohtaan jossa nimielementit ovat.
     * @return {@link PN} Henkilön nimi
     */
    public PN getPN(String xpath);
    
    /**
     * Palauttaa xpath:iin osuneiden nodejen määrän
     * 
     * @param expression xpath expression jolla nodet haetaan.
     * @return nodejen lukumäärä
     */
    public int getNodeCount(String expression);


    /**
     * Palauttaa listan NodeInfo-objekteja xpath:in perusteella.
     * 
     * @param expression xpath expression jolla NodeInfot haetaan.
     * @return lista NodeInfo-objekteja
     */
    public List<NodeInfo> getNodeList(String expression);
}
