package fi.kela.kanta.ptayhteiset.liiketoimintalogiikka.util;

import com.google.gson.internal.LinkedTreeMap;
import fi.kela.kanta.ptayhteiset.jwt.JWTContentDTO;
import org.apache.commons.lang3.ArrayUtils;
import org.apache.commons.lang3.StringUtils;

import java.util.Arrays;
import java.util.Map;
import java.util.Objects;

import static fi.kela.kanta.ptayhteiset.liiketoimintalogiikka.util.LiiketoimintaHelper.validatePatientId;
import static fi.kela.kanta.ptayhteiset.liiketoimintalogiikka.validaattorit.YTunnusValidator.validoiYTunnusMuoto;
import static fi.kela.kanta.ptayhteiset.tietomalli.vakiot.PTAYhteisetConstants.*;
import static org.apache.commons.lang3.ArrayUtils.isNotEmpty;
import static org.apache.commons.lang3.StringUtils.*;

/**
 * Kaikille kutsuille pakolliset kentät tarkistetaan
 * allekirjoituspalvelussa requiredKeys-kohdassa
 *
 * Ehdollisesti pakolliset tarkistukset listataan tässä, varsinaiset poikkeukset ja
 * virheet nostetaan käyttävissä palveluissa
 *
 * Tarkastukset bundlea vasten tehdään käyttävässä palvelussa
 */
public class JWTHelper {

    public static boolean isValidHatahaku(JWTContentDTO jwtContentDTO) {
        return isValidCode(jwtContentDTO.specialReason, CODESYSTEM_KEY, SPECIAL_REASON_CODESYSTEM)
                && equalsIgnoreCase(jwtContentDTO.specialReason.get(CODE_KEY), HATAHAKU_SPECIAL_REASON_VALUE)
                && isValidPractitioner(jwtContentDTO);
    }


    public static boolean isValidYhteisliittymistilanne(JWTContentDTO jwtContentDTO) {
        return !equalsIgnoreCase(jwtContentDTO.subscriberId, jwtContentDTO.requesterId) &&
                isNoneBlank(jwtContentDTO.subscriberUnitId, jwtContentDTO.subscriberUnitName) &&
                isNoneBlank(jwtContentDTO.requesterUnitId, jwtContentDTO.requesterUnitName);
    }


    public static boolean isValidPractitioner(JWTContentDTO jwtContentDTO) {
        return isValidCode(jwtContentDTO.practitionerId, CODESYSTEM_KEY, PATIENTID_OFFICIAL_CODEBASE,
                TEMP_PATIENTID_OFFICIAL_CODEBASE)
                && isNoneBlank(jwtContentDTO.practitionerFamily)
                && isValidAuthenticationMethod(jwtContentDTO.authenticationMethod)
                && isNotEmpty(jwtContentDTO.practitionerGiven)
                && isValidSsn(jwtContentDTO.practitionerId, PATIENTID_OFFICIAL_CODEBASE);
    }


    public static boolean isValidCitizen(JWTContentDTO jwtContentDTO) {
        return isValidCode(jwtContentDTO.citizenId, CODESYSTEM_KEY, PATIENTID_OFFICIAL_CODEBASE,
                TEMP_PATIENTID_OFFICIAL_CODEBASE)
                && isNoneBlank(jwtContentDTO.citizenFamily)
                && isValidAuthenticationMethod(jwtContentDTO.authenticationMethod)
                && isNotEmpty(jwtContentDTO.citizenGiven)
                && isValidSsn(jwtContentDTO.citizenId, PATIENTID_OFFICIAL_CODEBASE);
    }


    public static boolean isValidRequestedRecord(JWTContentDTO jwtContentDTO) {
        return isValidCode(jwtContentDTO.requestedRecord, CODESYSTEM_KEY, PATIENTID_OFFICIAL_CODEBASE,
                TEMP_PATIENTID_OFFICIAL_CODEBASE)
                && isValidSsn(jwtContentDTO.requestedRecord, PATIENTID_OFFICIAL_CODEBASE);
    }


    public static boolean isValidRegister(JWTContentDTO jwtContentDTO) {
        return isValidCode(jwtContentDTO.register, CODESYSTEM_KEY, REKISTERI_CODESYSTEM);
    }


    private static boolean isValidSsn(Map<String, String> value, String codeSystem) {
        return (!equalsIgnoreCase(value.get(CODESYSTEM_KEY), codeSystem)
                || (equalsIgnoreCase(value.get(CODESYSTEM_KEY), codeSystem)
                && validatePatientId(value.get(VALUE_KEY))));
    }


    private static boolean isValidAuthenticationMethod(Map<String, String> value) {
        return isValidCode(value, CODESYSTEM_KEY, AUTHENTICATION_METHOD_CODESYSTEM);
    }


    public static boolean isValidRegisterSpecifier(JWTContentDTO jwtContentDTO) {
        Map<String, String> registerSpecifier = getRegisterSpecifier(jwtContentDTO);

        return isValidRegister(jwtContentDTO)
                && isNoneBlank(jwtContentDTO.requesterCustodian, jwtContentDTO.requesterCustodianName)
                && equalsIgnoreCase(TYOTERVEYS_REGISTER_SPECIFIER, jwtContentDTO.register.get(CODE_KEY))
                && isValidCode(registerSpecifier, CODESYSTEM_KEY, VALID_REGISTER_SPECIFIER_CODESYSTEMS)
                && isValidSpecifierValue(registerSpecifier);
    }


    private static boolean isValidSpecifierValue(Map<String, String> value) {

        String codeSystem = value.get(CODESYSTEM_KEY);

        switch (codeSystem) {

            case OTHER_REGISTER_SPECIFIER_CODESYSTEM:
                return true;

            case PATIENTID_OFFICIAL_CODEBASE:
                return isValidSsn(value, PATIENTID_OFFICIAL_CODEBASE);

            case REGISTER_SPECIFIER_CODESYSTEM:
                return isValidCompany(value);

            default:
                return false;

        }
    }


    private static Map<String, String> getRegisterSpecifier(JWTContentDTO jwtContentDTO) {
        if (!hasCombinedSystemValue(jwtContentDTO.registerSpecifier)) {
            return jwtContentDTO.registerSpecifier;
        }

        Map<String, String> registerSpecifier = splitCombinedSystemValueMap(jwtContentDTO.registerSpecifier,
                VALID_REGISTER_SPECIFIER_CODESYSTEMS);
        if (!isBusinessIdNormalized(registerSpecifier))
            return null;

        return registerSpecifier;
    }


    private static boolean hasCombinedSystemValue(Map<String, String> map) {
        return map != null && map.get(VALUE_KEY) == null;
    }


    private static String getSystemFromCombinedSystemValue(Map<String, String> map, String... codeSystem) {
        if (map == null || StringUtils.isBlank(map.get(CODESYSTEM_KEY))) {
            return null;
        }

        String value = map.get(CODESYSTEM_KEY);
        return Arrays.stream(codeSystem)
                .filter(system -> value != null && value.startsWith(system))
                .findFirst()
                .orElse(null);
    }


    private static String getValueFromCombinedSystemValue(Map<String, String> map, String... codeSystem) {
        if (map == null || StringUtils.isBlank(map.get(CODESYSTEM_KEY))) {
            return null;
        }

        String system = getSystemFromCombinedSystemValue(map, codeSystem);
        return system != null ? StringUtils.removeStart(map.get(CODESYSTEM_KEY), system + SEPARATOR) : null;
    }


    private static Map<String, String> splitCombinedSystemValueMap(Map<String, String> map, String... codeSystem) {
        Map<String, String> separatedMap = new LinkedTreeMap<>();
        separatedMap.put(CODESYSTEM_KEY, getSystemFromCombinedSystemValue(map, codeSystem));
        separatedMap.put(VALUE_KEY, getValueFromCombinedSystemValue(map, codeSystem));
        return separatedMap;
    }


    private static boolean isBusinessIdNormalized(Map<String, String> map) {
        if (map == null) {
            return false;
        }
        if (!REGISTER_SPECIFIER_CODESYSTEM.equals(map.get(CODESYSTEM_KEY))) {
            return true;
        }

        String value = map.get(VALUE_KEY);
        return value != null && !value.startsWith("0") && !value.contains("-");
    }


    public static boolean isValidPuolestaAsiointi(JWTContentDTO jwtContentDTO) {
        return isValidCitizen(jwtContentDTO)
                && isValidRequestedRecord(jwtContentDTO)
                && !Objects.equals(jwtContentDTO.citizenId, jwtContentDTO.requestedRecord);
    }


    public static boolean isValidUsageSituation(JWTContentDTO jwtContentDTO) {
        return isValidCode(jwtContentDTO.usageSituation, CODESYSTEM_KEY, USAGE_SITUATION_CODESYSTEM);
    }


    public static boolean isValidServiceEventId(JWTContentDTO jwtContentDTO) {
        return isNotBlank(jwtContentDTO.serviceEventId);
    }


    private static boolean isValidCompany(Map<String, String> value) {
        return validoiYTunnusMuoto(value.get(VALUE_KEY));
    }


    private static boolean isValidCode(Map<String, String> map, String key, String... codeSystem) {
        return map != null && map.containsKey(key) && ArrayUtils.contains(codeSystem,
                map.get(key)) && StringUtils.isNotBlank(map.get(key));
    }


    private JWTHelper() {
        // private constructor
    }
}
