package fi.kela.kanta.ptayhteiset.liiketoimintalogiikka.util;

import java.util.ArrayList;
import java.util.Arrays;
import java.util.List;
import java.util.regex.Pattern;
import java.util.stream.Collectors;

import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import fi.kela.kanta.ptayhteiset.tietomalli.poikkeukset.AppException;
import fi.kela.kanta.ptayhteiset.tietomalli.poikkeukset.ErrorMessages;
import fi.kela.kanta.ptayhteiset.tietomalli.vakiot.PTAYhteisetConstants;

/**
 * Apuluokka merkkijonotietojen käsittelyyn
 */
public class TietomallinKasittelyHelper {

    private static final String VIRHEELLINEN_PARAMETRI = "Virheellinen parametri";

    private static final Logger LOG = LoggerFactory.getLogger(TietomallinKasittelyHelper.class);


    /**
     * Muodostaa listan arvoista erotinmerkillä yhdistetyn merkkijonon. Jos suorituksessa tapahtuu virhe, heitetään
     * {@link AppException}
     * 
     * @param values lista
     * @param separator erotinmerkki
     * @return muodostettu merkkijono, tai null mikäli annettu lista oli null tai tyhjä
     */
    public static String formatMultiValue(List<String> values, String separator) {

	LOG.trace(LogUtils.LOG_BEGIN);

	String retVal = null;

	if (separator == null) {
	    LOG.error(VIRHEELLINEN_PARAMETRI.concat(PTAYhteisetConstants.FORMATTING_ANCHOR), ErrorMessages.ERROR_CODE_1017);
	    throw new AppException(VIRHEELLINEN_PARAMETRI, ErrorMessages.ERROR_CODE_1017);
	}

	if (values != null && !values.isEmpty()) {
	    retVal = String.join(separator, values).concat(separator);
	}

	LOG.trace(LogUtils.LOG_END);

	return retVal;
    }


    /**
     * Muodostaa erotinmerkillä yhdistetystä merkkijonosta merkkijonolistan. Jos suorituksessa tapahtuu virhe, heitetään
     * {@link AppException}
     * 
     * @param value erotinmerkillä yhdistetty merkkijono
     * @param separator erotinmerkki
     * @return muodostettu merkkijonolista
     */
    public static List<String> convertMultiValueStringToList(String value, String separator) {

	LOG.trace(LogUtils.LOG_BEGIN);

	List<String> retList = new ArrayList<>();

	if (separator == null) {
	    LOG.error(VIRHEELLINEN_PARAMETRI.concat(PTAYhteisetConstants.FORMATTING_ANCHOR), ErrorMessages.ERROR_CODE_1017);
	    throw new AppException(VIRHEELLINEN_PARAMETRI, ErrorMessages.ERROR_CODE_1017);
	}

	if (value != null && !value.isEmpty()) {
	    Pattern p = Pattern.compile(Pattern.quote(separator));
	    retList = Arrays.asList(p.split(value));
	}

	LOG.trace(LogUtils.LOG_END);

	return retList;
    }


    /**
     * Poistaa listasta duplikaattiarvot ja palauttaa muokatun listan
     * 
     * @param origList lista josta poistetaan duplikaattiarvot
     * @return lista josta on poistettu duplikaattiarvot
     */
    public static List<String> removeDuplicates(List<String> origList) {
	return origList.stream().distinct().collect(Collectors.toList());
    }


    /**
     * Vertaa vastaako merkkijonojen arvot toisiaan.
     * 
     * @param firstStr vertailtava arvo
     * @param secondStr vertailtava arvo
     * @return true jos arvot vastaavat toisiaan, muuten false.
     */
    public static boolean compareStringValues(String firstStr, String secondStr) {

	LOG.trace(LogUtils.LOG_BEGIN);

	boolean isEquals = false;

	String firstValue = "";
	String secondValue = "";

	if (firstStr != null && !firstStr.isEmpty()) {
	    firstValue = firstStr;
	}

	if (secondStr != null && !secondStr.isEmpty()) {
	    secondValue = secondStr;
	}

	if (firstValue.equals(secondValue)) {
	    isEquals = true;
	}

	LOG.trace(LogUtils.LOG_END);

	return isEquals;
    }



    /**
     * Privaatti konstruktori
     */
    private  TietomallinKasittelyHelper() {
	throw new IllegalStateException("Utility class");
    }
}
